'use client'

import { useState, useEffect } from 'react'
import { supabase } from '@/lib/supabase-client'
import { Market } from '@/lib/types'

export function useMarkets(limit: number = 20) {
  const [markets, setMarkets] = useState<Market[]>([])
  const [loading, setLoading] = useState(true)
  const [error, setError] = useState<string | null>(null)

  useEffect(() => {
    const fetchMarkets = async (isBackground = false) => {
      try {
        // Only show loading on initial fetch, not background updates
        if (!isBackground) {
          setLoading(true)
          setError(null)
        }

        const { data: marketsData, error: marketsError } = await supabase
          .from('markets')
          .select('*')
          .order('created_at', { ascending: false })
          .limit(limit)

        if (marketsError) throw marketsError

        // Calculate pool information for each market
        const marketsWithPools = await Promise.all(
          (marketsData || []).map(async (market) => {
            if (market.market_type === 'multi_outcome') {
              // Handle multi-outcome markets
              const { data: outcomes } = await supabase
                .from('market_outcomes')
                .select('*')
                .eq('market_id', market.id)
                .order('created_at', { ascending: true })

              const { data: bets } = await supabase
                .from('bets')
                .select('amount, outcome, outcome_id')
                .eq('market_id', market.id)
                .eq('status', 'matched')

              // Calculate total volume from bets
              const totalVolume = bets?.reduce((sum: number, bet: any) => sum + (bet.amount || 0), 0) || 0

              // Update outcomes with calculated prices
              const updatedOutcomes = outcomes?.map(outcome => {
                const outcomeBets = bets?.filter((bet: any) => bet.outcome_id === outcome.id) || []
                const outcomeVolume = outcomeBets.reduce((sum: number, bet: any) => sum + (bet.amount || 0), 0)
                
                return {
                  ...outcome,
                  volume: outcomeVolume,
                  price: market.total_liquidity > 0 ? outcome.pool / market.total_liquidity : 0
                }
              }) || []

              return {
                ...market,
                outcomes: updatedOutcomes,
                total_volume: totalVolume,
                total_liquidity: market.total_liquidity || 0
              }
            } else {
              // Handle binary markets (backward compatibility)
              const { data: bets } = await supabase
                .from('bets')
                .select('seeded_amount, live_amount, outcome')
                .eq('market_id', market.id)
                .eq('status', 'matched')

              // Calculate pools for each outcome and type
              let yesSeededPool = 0
              let noSeededPool = 0
              let yesLivePool = 0
              let noLivePool = 0
              let totalSeededVolume = 0
              let totalLiveVolume = 0

              bets?.forEach(bet => {
                const seededAmount = bet.seeded_amount || 0
                const liveAmount = bet.live_amount || 0
                
                totalSeededVolume += seededAmount
                totalLiveVolume += liveAmount
                
                if (bet.outcome === 'yes') {
                  yesSeededPool += seededAmount
                  yesLivePool += liveAmount
                } else {
                  noSeededPool += seededAmount
                  noLivePool += liveAmount
                }
              })

              // Calculate prices based on pools
              const totalYesPool = yesSeededPool + yesLivePool
              const totalNoPool = noSeededPool + noLivePool
              const totalPool = totalYesPool + totalNoPool

              // Calculate prices as percentages of total pool
              const yesPrice = totalPool > 0 ? totalYesPool / totalPool : 0.5
              const noPrice = totalPool > 0 ? totalNoPool / totalPool : 0.5

              return {
                ...market,
                yes_price: yesPrice,
                no_price: noPrice,
                yes_pool: totalYesPool,
                no_pool: totalNoPool,
                seeded_volume: totalSeededVolume,
                live_volume: totalLiveVolume,
                total_volume: totalSeededVolume + totalLiveVolume,
                total_liquidity: market.total_liquidity || totalPool
              }
            }
          })
        )

        setMarkets(marketsWithPools)
      } catch (err) {
        // Only set error on initial fetch, not background updates
        if (!isBackground) {
          setError(err instanceof Error ? err.message : 'Failed to fetch markets')
        }
        console.error('Error fetching markets:', err)
      } finally {
        // Only set loading to false on initial fetch
        if (!isBackground) {
          setLoading(false)
        }
      }
    }

    // Initial fetch (with loading state)
    fetchMarkets(false)

    // Set up background updates every 5 seconds (without loading state)
    const interval = setInterval(() => {
      fetchMarkets(true)
    }, 5000)

    // Cleanup interval on unmount
    return () => clearInterval(interval)
  }, [limit])

  return { markets, loading, error }
}