import { useState, useEffect, useCallback } from 'react'
import { supabase } from '@/lib/supabase-client'

export interface Payout {
  id: string
  market_id: string
  market_title: string
  user_id: string
  username: string
  email: string
  payout_type: 'winner_seeded' | 'winner_live' | 'creator_earnings' | 'admin_fee' | 'refund'
  amount: number
  multiplier: number | null
  status: 'pending' | 'processing' | 'completed' | 'failed' | 'cancelled'
  processed_at: string | null
  created_at: string
  transaction_id: string | null
}

export interface PayoutSummary {
  total_payouts: number
  total_amount: number
  completed_payouts: number
  pending_payouts: number
  failed_payouts: number
  winner_count: number
  creator_earnings: number
  admin_fees: number
}

export interface MarketPayoutSummary {
  market_id: string
  market_title: string
  total_winners: number
  total_payouts: number
  creator_earnings: number
  admin_fee: number
  winner_details: Array<{
    user_id: string
    username: string
    email: string
    seeded_payout: number
    live_payout: number
    total_payout: number
    multiplier: number
  }>
}

export function usePayouts() {
  const [payouts, setPayouts] = useState<Payout[]>([])
  const [summary, setSummary] = useState<PayoutSummary>({
    total_payouts: 0,
    total_amount: 0,
    completed_payouts: 0,
    pending_payouts: 0,
    failed_payouts: 0,
    winner_count: 0,
    creator_earnings: 0,
    admin_fees: 0
  })
  const [marketSummaries, setMarketSummaries] = useState<MarketPayoutSummary[]>([])
  const [loading, setLoading] = useState(true)
  const [error, setError] = useState<string | null>(null)

  const fetchPayouts = useCallback(async () => {
    try {
      setLoading(true)
      setError(null)

      const { data: payoutsData, error: payoutsError } = await supabase
        .from('payouts')
        .select(`
          *,
          markets!inner(title),
          profiles!inner(username, email)
        `)
        .order('created_at', { ascending: false })

      if (payoutsError) {
        if (payoutsError.message.includes('relation "payouts" does not exist')) {
          setError('Payout system not set up. Please run the setup script first.')
          return
        }
        throw payoutsError
      }

      // Transform payout data
      const transformedPayouts = (payoutsData || []).map(payout => ({
        id: payout.id,
        market_id: payout.market_id,
        market_title: (payout.markets as any)?.title || 'Unknown Market',
        user_id: payout.user_id,
        username: (payout.profiles as any)?.username || 'Unknown User',
        email: (payout.profiles as any)?.email || 'Unknown Email',
        payout_type: payout.payout_type,
        amount: payout.amount,
        multiplier: payout.multiplier,
        status: payout.status,
        processed_at: payout.processed_at,
        created_at: payout.created_at,
        transaction_id: payout.transaction_id
      }))

      setPayouts(transformedPayouts)

      // Calculate summary statistics
      const summaryData: PayoutSummary = {
        total_payouts: transformedPayouts.length,
        total_amount: transformedPayouts.reduce((sum, p) => sum + p.amount, 0),
        completed_payouts: transformedPayouts.filter(p => p.status === 'completed').length,
        pending_payouts: transformedPayouts.filter(p => p.status === 'pending').length,
        failed_payouts: transformedPayouts.filter(p => p.status === 'failed').length,
        winner_count: transformedPayouts.filter(p => p.payout_type.includes('winner')).length,
        creator_earnings: transformedPayouts
          .filter(p => p.payout_type === 'creator_earnings')
          .reduce((sum, p) => sum + p.amount, 0),
        admin_fees: transformedPayouts
          .filter(p => p.payout_type === 'admin_fee')
          .reduce((sum, p) => sum + p.amount, 0)
      }

      setSummary(summaryData)

      // Generate market summaries
      await generateMarketSummaries(transformedPayouts)

    } catch (err) {
      setError(err instanceof Error ? err.message : 'Failed to fetch payout data')
      console.error('Error fetching payout data:', err)
    } finally {
      setLoading(false)
    }
  }, [])

  const generateMarketSummaries = useCallback(async (payoutsData: Payout[]) => {
    try {
      // Get unique market IDs
      const marketIds = [...new Set(payoutsData.map(p => p.market_id))]
      
      const marketSummaries: MarketPayoutSummary[] = []
      
      for (const marketId of marketIds) {
        const marketPayouts = payoutsData.filter(p => p.market_id === marketId)
        const marketTitle = marketPayouts[0]?.market_title || 'Unknown Market'
        
        const winnerDetails = marketPayouts
          .filter(p => p.payout_type.includes('winner'))
          .reduce((acc: any[], payout) => {
            const existing = acc.find(w => w.user_id === payout.user_id)
            if (existing) {
              if (payout.payout_type === 'winner_seeded') {
                existing.seeded_payout = payout.amount
              } else if (payout.payout_type === 'winner_live') {
                existing.live_payout = payout.amount
              }
              existing.total_payout = existing.seeded_payout + existing.live_payout
            } else {
              acc.push({
                user_id: payout.user_id,
                username: payout.username,
                email: payout.email,
                seeded_payout: payout.payout_type === 'winner_seeded' ? payout.amount : 0,
                live_payout: payout.payout_type === 'winner_live' ? payout.amount : 0,
                total_payout: payout.amount,
                multiplier: payout.multiplier || 0
              })
            }
            return acc
          }, [])

        const summary: MarketPayoutSummary = {
          market_id: marketId,
          market_title: marketTitle,
          total_winners: winnerDetails.length,
          total_payouts: marketPayouts.reduce((sum, p) => sum + p.amount, 0),
          creator_earnings: marketPayouts.find(p => p.payout_type === 'creator_earnings')?.amount || 0,
          admin_fee: marketPayouts.find(p => p.payout_type === 'admin_fee')?.amount || 0,
          winner_details: winnerDetails.sort((a, b) => b.total_payout - a.total_payout)
        }

        marketSummaries.push(summary)
      }

      setMarketSummaries(marketSummaries.sort((a, b) => b.total_payouts - a.total_payouts))

    } catch (err) {
      console.error('Error generating market summaries:', err)
    }
  }, [])

  const triggerMarketPayouts = useCallback(async (marketId: string, outcome: 'yes' | 'no') => {
    try {
      setError(null)

      const { data, error } = await supabase
        .rpc('trigger_market_payouts', {
          p_market_id: marketId,
          p_outcome: outcome
        })

      if (error) {
        throw error
      }

      const result = data?.[0]
      if (!result || !result.success) {
        throw new Error(result?.message || 'Failed to trigger payouts')
      }

      // Refresh payout data
      await fetchPayouts()

      return {
        success: true,
        message: result.message,
        totalPayouts: result.total_payouts,
        totalAmount: result.total_amount,
        winnerCount: result.winner_count
      }

    } catch (err) {
      const errorMessage = err instanceof Error ? err.message : 'Failed to trigger payouts'
      setError(errorMessage)
      console.error('Error triggering payouts:', err)
      return {
        success: false,
        error: errorMessage
      }
    }
  }, [fetchPayouts])

  const getMarketPayoutSummary = useCallback(async (marketId: string) => {
    try {
      const { data, error } = await supabase
        .rpc('get_market_payout_summary', {
          p_market_id: marketId
        })

      if (error) {
        throw error
      }

      return data?.[0] || null

    } catch (err) {
      console.error('Error fetching market payout summary:', err)
      return null
    }
  }, [])

  useEffect(() => {
    fetchPayouts()
  }, [fetchPayouts])

  return {
    payouts,
    summary,
    marketSummaries,
    loading,
    error,
    refetch: fetchPayouts,
    triggerMarketPayouts,
    getMarketPayoutSummary
  }
}

export function usePayoutSummary(marketId?: string) {
  const [summary, setSummary] = useState<any>(null)
  const [loading, setLoading] = useState(false)
  const [error, setError] = useState<string | null>(null)

  const fetchSummary = useCallback(async () => {
    if (!marketId) return

    try {
      setLoading(true)
      setError(null)

      const { data, error } = await supabase
        .rpc('get_market_payout_summary', {
          p_market_id: marketId
        })

      if (error) {
        throw error
      }

      setSummary(data?.[0] || null)

    } catch (err) {
      setError(err instanceof Error ? err.message : 'Failed to fetch payout summary')
      console.error('Error fetching payout summary:', err)
    } finally {
      setLoading(false)
    }
  }, [marketId])

  useEffect(() => {
    fetchSummary()
  }, [fetchSummary])

  return {
    summary,
    loading,
    error,
    refetch: fetchSummary
  }
}
