import { useEffect, useRef, useState } from 'react'

interface PerformanceMetrics {
  renderTime: number
  memoryUsage?: number
  componentName: string
  timestamp: number
}

export function usePerformanceMonitor(componentName: string, enabled: boolean = true) {
  const renderStartTime = useRef<number>(0)
  const renderCount = useRef<number>(0)

  useEffect(() => {
    if (!enabled || process.env.NODE_ENV === 'production') return

    renderStartTime.current = performance.now()
    renderCount.current += 1

    return () => {
      const renderTime = performance.now() - renderStartTime.current
      
      // Only log if render time is significant (> 16ms for 60fps)
      if (renderTime > 16) {
        const metrics: PerformanceMetrics = {
          renderTime,
          componentName,
          timestamp: Date.now(),
          memoryUsage: (performance as any).memory?.usedJSHeapSize
        }

        console.warn(`Slow render detected in ${componentName}:`, {
          renderTime: `${renderTime.toFixed(2)}ms`,
          renderCount: renderCount.current,
          memoryUsage: metrics.memoryUsage ? `${(metrics.memoryUsage / 1024 / 1024).toFixed(2)}MB` : 'N/A'
        })
      }
    }
  })

  // Monitor memory usage
  useEffect(() => {
    if (!enabled || process.env.NODE_ENV === 'production') return

    const checkMemoryUsage = () => {
      if ((performance as any).memory) {
        const memory = (performance as any).memory
        const usedMB = memory.usedJSHeapSize / 1024 / 1024
        const limitMB = memory.jsHeapSizeLimit / 1024 / 1024

        // Warn if memory usage is high (> 80% of limit)
        if (usedMB / limitMB > 0.8) {
          console.warn(`High memory usage detected:`, {
            used: `${usedMB.toFixed(2)}MB`,
            limit: `${limitMB.toFixed(2)}MB`,
            percentage: `${((usedMB / limitMB) * 100).toFixed(1)}%`
          })
        }
      }
    }

    const interval = setInterval(checkMemoryUsage, 30000) // Check every 30 seconds
    return () => clearInterval(interval)
  }, [enabled])
}

export function useDebounce<T>(value: T, delay: number): T {
  const [debouncedValue, setDebouncedValue] = useState<T>(value)

  useEffect(() => {
    const handler = setTimeout(() => {
      setDebouncedValue(value)
    }, delay)

    return () => {
      clearTimeout(handler)
    }
  }, [value, delay])

  return debouncedValue
}

export function useThrottle<T>(value: T, limit: number): T {
  const [throttledValue, setThrottledValue] = useState<T>(value)
  const lastRan = useRef<number>(Date.now())

  useEffect(() => {
    const handler = setTimeout(() => {
      if (Date.now() - lastRan.current >= limit) {
        setThrottledValue(value)
        lastRan.current = Date.now()
      }
    }, limit - (Date.now() - lastRan.current))

    return () => {
      clearTimeout(handler)
    }
  }, [value, limit])

  return throttledValue
}
