'use client'

import { useState, useEffect } from 'react'
import { supabase } from '@/lib/supabase-client'
import { Profile, WalletTransaction, Bet } from '@/lib/types'

interface BetWithOutcome extends Bet {
  potentialWinnings?: number
  currentPrice?: number
  poolShare?: number
}

interface PortfolioData {
  profile: Profile | null
  transactions: WalletTransaction[]
  bets: BetWithOutcome[]
  totalProfit: number
  activeBets: number
  winRate: number
  totalPotentialWinnings: number
  yesPotentialWinnings: number
  noPotentialWinnings: number
}

export function usePortfolio(userId?: string) {
  const [portfolio, setPortfolio] = useState<PortfolioData>({
    profile: null,
    transactions: [],
    bets: [],
    totalProfit: 0,
    activeBets: 0,
    winRate: 0,
    totalPotentialWinnings: 0,
    yesPotentialWinnings: 0,
    noPotentialWinnings: 0
  })
  const [loading, setLoading] = useState(true)
  const [error, setError] = useState<string | null>(null)

  useEffect(() => {
    const fetchPortfolio = async () => {
      if (!userId) {
        setLoading(false)
        return
      }

      try {
        setLoading(true)
        setError(null)

        // Fetch user profile
        const { data: profile, error: profileError } = await supabase
          .from('profiles')
          .select('*')
          .eq('id', userId)
          .single()

        if (profileError) throw profileError

        // Fetch all transactions from unified table (including all types: deposits, withdrawals, bets, wins, seeds, liquidations, etc.)
        const { data: transactions, error: transactionsError } = await supabase
          .from('unified_transactions')
          .select(`
            *,
            markets (
              id,
              title
            )
          `)
          .eq('user_id', userId)
          .order('created_at', { ascending: false })
          .limit(50)

        if (transactionsError) console.warn('Failed to fetch transactions:', transactionsError)

        // Fetch user bets with market info including pool data
        const { data: bets, error: betsError } = await supabase
          .from('bets')
          .select(`
            *,
            markets (
              id,
              title,
              status,
              outcome,
              end_date,
              yes_pool,
              no_pool,
              total_volume,
              total_liquidity,
              yes_price,
              no_price
            )
          `)
          .eq('user_id', userId)
          .order('created_at', { ascending: false })

        if (betsError) console.warn('Failed to fetch bets:', betsError)

        // Calculate portfolio metrics and potential outcomes
        const betsWithOutcomes = (bets || []).map((bet: any) => {
          const market = bet.markets
          let potentialWinnings = 0
          let currentPrice = 0
          let poolShare = 0

          if (bet.status === 'seeded' || bet.status === 'live' || bet.status === 'matched') {
            if (market && market.yes_pool !== null && market.no_pool !== null) {
              // Calculate potential winnings based on current pool sizes
              if (bet.outcome === 'yes' && market.yes_pool > 0) {
                currentPrice = market.yes_price || 0
                poolShare = (bet.amount || 0) / market.yes_pool
                // If YES wins, user gets their share of the total pool
                const totalPool = (market.yes_pool || 0) + (market.no_pool || 0)
                potentialWinnings = totalPool * poolShare
              } else if (bet.outcome === 'no' && market.no_pool > 0) {
                currentPrice = market.no_price || 0
                poolShare = (bet.amount || 0) / market.no_pool
                // If NO wins, user gets their share of the total pool
                const totalPool = (market.yes_pool || 0) + (market.no_pool || 0)
                potentialWinnings = totalPool * poolShare
              }
            } else {
              // Fallback calculation if market data is missing
              potentialWinnings = (bet.amount || 0) * 2 // Assume 1:1 payout ratio
              currentPrice = bet.outcome === 'yes' ? 0.5 : 0.5
            }
          } else if (bet.status === 'won') {
            // Already won bets show their actual payout
            potentialWinnings = bet.potential_payout || 0
          } else if (bet.status === 'lost') {
            // Lost bets show 0 potential winnings
            potentialWinnings = 0
          }

          return {
            ...bet,
            potentialWinnings,
            currentPrice,
            poolShare
          }
        })

        const totalProfit = betsWithOutcomes.reduce((sum: number, bet: any) => {
          if (bet.status === 'won') {
            return sum + (bet.potential_payout || 0)
          } else if (bet.status === 'lost') {
            return sum - (bet.amount || 0)
          }
          return sum
        }, 0)

        const totalPotentialWinnings = betsWithOutcomes.reduce((sum: number, bet: any) => {
          if (bet.status === 'seeded' || bet.status === 'live' || bet.status === 'matched') {
            return sum + bet.potentialWinnings
          }
          return sum
        }, 0)

        const yesPotentialWinnings = betsWithOutcomes.reduce((sum: number, bet: any) => {
          if ((bet.status === 'seeded' || bet.status === 'live' || bet.status === 'matched') && bet.outcome === 'yes') {
            return sum + bet.potentialWinnings
          }
          return sum
        }, 0)

        const noPotentialWinnings = betsWithOutcomes.reduce((sum: number, bet: any) => {
          if ((bet.status === 'seeded' || bet.status === 'live' || bet.status === 'matched') && bet.outcome === 'no') {
            return sum + bet.potentialWinnings
          }
          return sum
        }, 0)

        const activeBets = betsWithOutcomes.filter((bet: any) => 
          bet.status === 'seeded' || bet.status === 'live' || bet.status === 'matched'
        ).length

        const wonBets = betsWithOutcomes.filter((bet: any) => bet.status === 'won').length
        const totalBets = betsWithOutcomes.filter((bet: any) => 
          bet.status === 'won' || bet.status === 'lost'
        ).length
        const winRate = totalBets > 0 ? Math.round((wonBets / totalBets) * 100) : 0

        setPortfolio({
          profile: profile as Profile,
          transactions: (transactions || []) as WalletTransaction[],
          bets: betsWithOutcomes as BetWithOutcome[],
          totalProfit,
          activeBets,
          winRate,
          totalPotentialWinnings,
          yesPotentialWinnings,
          noPotentialWinnings
        })
      } catch (err) {
        setError(err instanceof Error ? err.message : 'Failed to fetch portfolio')
        console.error('Error fetching portfolio:', err)
      } finally {
        setLoading(false)
      }
    }

    fetchPortfolio()

    // Set up real-time subscriptions for live updates
    const profileSubscription = supabase
      .channel('portfolio-profile-changes')
      .on(
        'postgres_changes',
        {
          event: 'UPDATE',
          schema: 'public',
          table: 'profiles',
          filter: `id=eq.${userId}`
        },
        (payload) => {
          console.log('Profile updated:', payload)
          setPortfolio(prev => ({
            ...prev,
            profile: payload.new as Profile
          }))
        }
      )
      .subscribe()

    const transactionsSubscription = supabase
      .channel('portfolio-transactions-changes')
      .on(
        'postgres_changes',
        {
          event: '*',
          schema: 'public',
          table: 'unified_transactions',
          filter: `user_id=eq.${userId}`
        },
        (payload) => {
          console.log('Transaction updated:', payload)
          // Refetch portfolio to get updated calculations
          fetchPortfolio()
        }
      )
      .subscribe()

    const betsSubscription = supabase
      .channel('portfolio-bets-changes')
      .on(
        'postgres_changes',
        {
          event: '*',
          schema: 'public',
          table: 'bets',
          filter: `user_id=eq.${userId}`
        },
        (payload) => {
          console.log('Bet updated:', payload)
          // Refetch portfolio to get updated calculations
          fetchPortfolio()
        }
      )
      .subscribe()

    const marketsSubscription = supabase
      .channel('portfolio-markets-changes')
      .on(
        'postgres_changes',
        {
          event: 'UPDATE',
          schema: 'public',
          table: 'markets'
        },
        (payload) => {
          console.log('Market updated:', payload)
          // Refetch portfolio to get updated market data and calculations
          fetchPortfolio()
        }
      )
      .subscribe()

    // Cleanup subscriptions on unmount
    return () => {
      profileSubscription.unsubscribe()
      transactionsSubscription.unsubscribe()
      betsSubscription.unsubscribe()
      marketsSubscription.unsubscribe()
    }
  }, [userId])

  return { portfolio, loading, error }
}