import { useState, useEffect, useCallback, useMemo } from 'react'
import { supabase } from '@/lib/supabase-client'
import { Profile, BetWithMarket, TransactionWithMarket } from '@/lib/types'

interface PortfolioData {
  profile: Profile | null
  transactions: TransactionWithMarket[]
  bets: BetWithMarket[]
  totalProfit: number
  activeBets: number
  winRate: number
  totalPotentialWinnings: number
  yesPotentialWinnings: number
  noPotentialWinnings: number
}

export function usePortfolioOptimized(userId?: string) {
  const [portfolio, setPortfolio] = useState<PortfolioData>({
    profile: null,
    transactions: [],
    bets: [],
    totalProfit: 0,
    activeBets: 0,
    winRate: 0,
    totalPotentialWinnings: 0,
    yesPotentialWinnings: 0,
    noPotentialWinnings: 0
  })
  const [loading, setLoading] = useState(true)
  const [error, setError] = useState<string | null>(null)
  const [lastFetch, setLastFetch] = useState<number>(0)

  // Debounced fetch function to prevent excessive calls
  const debouncedFetch = useCallback(
    (() => {
      let timeoutId: NodeJS.Timeout
      return (force = false) => {
        const now = Date.now()
        // Only fetch if it's been more than 2 seconds since last fetch or if forced
        if (force || now - lastFetch > 2000) {
          clearTimeout(timeoutId)
          timeoutId = setTimeout(() => {
            fetchPortfolio(force)
            setLastFetch(Date.now())
          }, 300) // 300ms debounce
        }
      }
    })(),
    [lastFetch]
  )

  const fetchPortfolio = useCallback(async (isBackground = false) => {
    if (!userId) {
      setLoading(false)
      return
    }

    try {
      if (!isBackground) {
        setLoading(true)
        setError(null)
      }

      // Fetch data in parallel with optimized queries
      const [profileResult, transactionsResult, betsResult] = await Promise.allSettled([
        // Profile with minimal data
        supabase
          .from('profiles')
          .select('id, username, email, balance, kyc_verified, created_at')
          .eq('id', userId)
          .single(),

        // Transactions with pagination
        supabase
          .from('unified_transactions')
          .select(`
            id,
            type,
            amount,
            status,
            created_at,
            market_id,
            markets!inner(id, title)
          `)
          .eq('user_id', userId)
          .order('created_at', { ascending: false })
          .limit(20), // Reduced from 50 to 20

        // Bets with minimal market data
        supabase
          .from('bets')
          .select(`
            id,
            amount,
            outcome,
            status,
            created_at,
            potential_payout,
            market_id,
            markets!inner(id, title, status, outcome, yes_price, no_price)
          `)
          .eq('user_id', userId)
          .order('created_at', { ascending: false })
          .limit(50)
      ])

      // Process results safely
      const profile = profileResult.status === 'fulfilled' && profileResult.value.data 
        ? profileResult.value.data as Profile
        : null

      const transactions = transactionsResult.status === 'fulfilled' && transactionsResult.value.data 
        ? (transactionsResult.value.data as unknown) as TransactionWithMarket[]
        : []

      const bets = betsResult.status === 'fulfilled' && betsResult.value.data 
        ? betsResult.value.data as BetWithMarket[]
        : []

      // Calculate stats efficiently
      const activeBets = bets.filter(bet => bet.status === 'matched').length
      const wonBets = bets.filter(bet => bet.status === 'won').length
      const winRate = bets.length > 0 ? (wonBets / bets.length) * 100 : 0

      // Calculate potential winnings more efficiently
      const { totalPotentialWinnings, yesPotentialWinnings, noPotentialWinnings } = bets.reduce(
        (acc, bet) => {
          if (bet.status === 'matched' && bet.potential_payout) {
            acc.totalPotentialWinnings += bet.potential_payout
            if (bet.outcome === 'yes') {
              acc.yesPotentialWinnings += bet.potential_payout
            } else {
              acc.noPotentialWinnings += bet.potential_payout
            }
          }
          return acc
        },
        { totalPotentialWinnings: 0, yesPotentialWinnings: 0, noPotentialWinnings: 0 }
      )

      // Calculate total profit
      const totalProfit = bets.reduce((sum, bet) => {
        if (bet.status === 'won' && bet.potential_payout) {
          return sum + (bet.potential_payout - bet.amount)
        } else if (bet.status === 'lost') {
          return sum - bet.amount
        }
        return sum
      }, 0)

      setPortfolio({
        profile,
        transactions,
        bets,
        totalProfit,
        activeBets,
        winRate,
        totalPotentialWinnings,
        yesPotentialWinnings,
        noPotentialWinnings
      })

    } catch (err) {
      console.error('Error fetching portfolio:', err)
      if (!isBackground) {
        setError(err instanceof Error ? err.message : 'Failed to fetch portfolio')
      }
    } finally {
      if (!isBackground) {
        setLoading(false)
      }
    }
  }, [userId])

  useEffect(() => {
    fetchPortfolio()

    // Set up optimized real-time subscriptions
    let subscriptions: any[] = []

    if (userId) {
      // Profile subscription - only for balance updates
      const profileSubscription = supabase
        .channel(`portfolio-profile-${userId}`)
        .on(
          'postgres_changes',
          {
            event: 'UPDATE',
            schema: 'public',
            table: 'profiles',
            filter: `id=eq.${userId}`
          },
          (payload) => {
            // Only update profile data, don't refetch everything
            setPortfolio(prev => ({
              ...prev,
              profile: payload.new as Profile
            }))
          }
        )
        .subscribe()

      // Transactions subscription - debounced refetch
      const transactionsSubscription = supabase
        .channel(`portfolio-transactions-${userId}`)
        .on(
          'postgres_changes',
          {
            event: 'INSERT',
            schema: 'public',
            table: 'unified_transactions',
            filter: `user_id=eq.${userId}`
          },
          () => {
            debouncedFetch()
          }
        )
        .subscribe()

      // Bets subscription - debounced refetch
      const betsSubscription = supabase
        .channel(`portfolio-bets-${userId}`)
        .on(
          'postgres_changes',
          {
            event: '*',
            schema: 'public',
            table: 'bets',
            filter: `user_id=eq.${userId}`
          },
          () => {
            debouncedFetch()
          }
        )
        .subscribe()

      subscriptions = [profileSubscription, transactionsSubscription, betsSubscription]
    }

    // Cleanup subscriptions on unmount
    return () => {
      subscriptions.forEach(sub => sub.unsubscribe())
    }
  }, [userId, fetchPortfolio, debouncedFetch])

  // Memoize the return value to prevent unnecessary re-renders
  return useMemo(() => ({
    portfolio,
    loading,
    error,
    refetch: () => fetchPortfolio(true)
  }), [portfolio, loading, error, fetchPortfolio])
}
