import { useState, useEffect, useCallback, useMemo } from 'react'
import { supabase } from '@/lib/supabase-client'
import { Profile } from '@/lib/types'

export function useProfileOptimized(userId?: string) {
  const [profile, setProfile] = useState<Profile | null>(null)
  const [loading, setLoading] = useState(true)
  const [error, setError] = useState<string | null>(null)
  const [lastFetch, setLastFetch] = useState<number>(0)

  const fetchProfile = useCallback(async (isBackground = false) => {
    console.log('useProfileOptimized: fetchProfile called', { userId, isBackground })
    if (!userId) {
      console.log('useProfileOptimized: No userId provided')
      setLoading(false)
      return
    }

    try {
      if (!isBackground) {
        setLoading(true)
        setError(null)
      }

      console.log('useProfileOptimized: Fetching profile from database...')
      const { data, error: profileError } = await supabase
        .from('profiles')
        .select('id, username, email, avatar_url, first_name, last_name, phone, country, date_of_birth, bio, kyc_verified, balance, created_at')
        .eq('id', userId)
        .single()

      if (profileError) {
        console.error('useProfileOptimized: Database error:', {
          error: profileError,
          message: profileError.message,
          details: profileError.details,
          hint: profileError.hint,
          code: profileError.code
        })
        throw profileError
      }

      console.log('useProfileOptimized: Profile data received:', data)
      setProfile(data as Profile)
      setLastFetch(Date.now())
    } catch (err) {
      console.error('Error fetching profile:', err)
      if (!isBackground) {
        setError(err instanceof Error ? err.message : 'Failed to fetch profile')
      }
    } finally {
      if (!isBackground) {
        setLoading(false)
      }
    }
  }, [userId])

  useEffect(() => {
    fetchProfile()

    // Set up real-time subscription for profile updates
    let subscription: any

    if (userId) {
      subscription = supabase
        .channel(`profile-${userId}`)
        .on(
          'postgres_changes',
          {
            event: 'UPDATE',
            schema: 'public',
            table: 'profiles',
            filter: `id=eq.${userId}`
          },
          (payload) => {
            setProfile(payload.new as Profile)
          }
        )
        .subscribe()
    }

    return () => {
      if (subscription) {
        subscription.unsubscribe()
      }
    }
  }, [userId, fetchProfile])

  const updateProfile = useCallback(async (updates: Partial<Profile>) => {
    if (!userId) return { success: false, error: 'No user ID' }

    try {
      setError(null)

      const { data, error } = await supabase
        .from('profiles')
        .update(updates)
        .eq('id', userId)
        .select('id, username, email, avatar_url, first_name, last_name, phone, country, date_of_birth, bio, kyc_verified, balance')
        .single()

      if (error) throw error

      setProfile(data as Profile)
      return { success: true, data }
    } catch (err) {
      const errorMessage = err instanceof Error ? err.message : 'Failed to update profile'
      setError(errorMessage)
      return { success: false, error: errorMessage }
    }
  }, [userId])

  // Memoize the return value
  return useMemo(() => ({
    profile,
    loading,
    error,
    updateProfile,
    refetch: () => fetchProfile(true)
  }), [profile, loading, error, updateProfile, fetchProfile])
}
