import { useState, useEffect } from 'react'
import { supabase } from '@/lib/supabase-client'

export interface RecentActivity {
  id: string
  type: 'bet' | 'deposit' | 'withdrawal' | 'market_created' | 'market_resolved' | 'payout' | 'seed' | 'liquidation'
  description: string
  amount?: number
  status?: string
  timestamp: string
  metadata?: any
  market_id?: string
  market_title?: string
}

export function useRecentActivity(userId?: string, limit: number = 20) {
  const [activities, setActivities] = useState<RecentActivity[]>([])
  const [loading, setLoading] = useState(true)
  const [error, setError] = useState<string | null>(null)

  useEffect(() => {
    if (!userId) {
      setLoading(false)
      return
    }

    fetchRecentActivity()

    // Set up real-time subscription
    const subscription = supabase
      .channel('recent_activity')
      .on('postgres_changes', 
        { 
          event: '*', 
          schema: 'public', 
          table: 'unified_transactions' 
        },
        () => {
          fetchRecentActivity()
        }
      )
      .on('postgres_changes',
        {
          event: '*',
          schema: 'public',
          table: 'bets'
        },
        () => {
          fetchRecentActivity()
        }
      )
      .subscribe()

    return () => {
      subscription.unsubscribe()
    }
  }, [userId, limit])

  const fetchRecentActivity = async () => {
    try {
      setLoading(true)
      setError(null)

      // Fetch recent transactions
      const { data: transactions, error: transactionsError } = await supabase
        .from('unified_transactions')
        .select(`
          id,
          type,
          amount,
          status,
          created_at,
          market_id,
          markets!inner(id, title)
        `)
        .eq('user_id', userId)
        .order('created_at', { ascending: false })
        .limit(limit)

      if (transactionsError) {
        throw transactionsError
      }

      // Fetch recent bets
      const { data: bets, error: betsError } = await supabase
        .from('bets')
        .select(`
          id,
          amount,
          outcome,
          status,
          created_at,
          market_id,
          markets!inner(id, title)
        `)
        .eq('user_id', userId)
        .order('created_at', { ascending: false })
        .limit(limit)

      if (betsError) {
        throw betsError
      }

      // Combine and format activities
      const allActivities: RecentActivity[] = []

      // Add transactions
      transactions?.forEach((transaction) => {
        allActivities.push({
          id: `transaction_${transaction.id}`,
          type: transaction.type as any,
          description: getTransactionDescription(transaction),
          amount: transaction.amount,
          status: transaction.status,
          timestamp: transaction.created_at,
          market_id: transaction.market_id,
          market_title: transaction.markets?.title,
          metadata: { transaction_id: transaction.id }
        })
      })

      // Add bets
      bets?.forEach((bet) => {
        allActivities.push({
          id: `bet_${bet.id}`,
          type: 'bet',
          description: `Placed ${bet.outcome} bet on "${bet.markets?.title}"`,
          amount: bet.amount,
          status: bet.status,
          timestamp: bet.created_at,
          market_id: bet.market_id,
          market_title: bet.markets?.title,
          metadata: { bet_id: bet.id, outcome: bet.outcome }
        })
      })

      // Sort by timestamp and limit
      allActivities.sort((a, b) => new Date(b.timestamp).getTime() - new Date(a.timestamp).getTime())
      setActivities(allActivities.slice(0, limit))

    } catch (err) {
      console.error('Error fetching recent activity:', err)
      setError(err instanceof Error ? err.message : 'Failed to fetch recent activity')
    } finally {
      setLoading(false)
    }
  }

  const getTransactionDescription = (transaction: any): string => {
    const amount = `$${transaction.amount?.toFixed(2) || '0.00'}`
    const market = transaction.markets?.title ? ` for "${transaction.markets.title}"` : ''
    
    switch (transaction.type) {
      case 'deposit':
        return `Deposited ${amount}${market}`
      case 'withdrawal':
        return `Withdrew ${amount}${market}`
      case 'seed':
        return `Seeded ${amount}${market}`
      case 'liquidation':
        return `Liquidation of ${amount}${market}`
      case 'payout':
        return `Received payout of ${amount}${market}`
      default:
        return `${transaction.type} of ${amount}${market}`
    }
  }

  const getActivityIcon = (type: string) => {
    switch (type) {
      case 'bet':
        return '🎯'
      case 'deposit':
        return '💰'
      case 'withdrawal':
        return '💸'
      case 'payout':
        return '🎉'
      case 'seed':
        return '🌱'
      case 'liquidation':
        return '💧'
      default:
        return '📊'
    }
  }

  const getActivityColor = (type: string) => {
    switch (type) {
      case 'bet':
        return 'text-blue-600 bg-blue-100'
      case 'deposit':
        return 'text-green-600 bg-green-100'
      case 'withdrawal':
        return 'text-red-600 bg-red-100'
      case 'payout':
        return 'text-purple-600 bg-purple-100'
      case 'seed':
        return 'text-yellow-600 bg-yellow-100'
      case 'liquidation':
        return 'text-orange-600 bg-orange-100'
      default:
        return 'text-gray-600 bg-gray-100'
    }
  }

  return {
    activities,
    loading,
    error,
    refetch: fetchRecentActivity,
    getActivityIcon,
    getActivityColor
  }
}
