'use client'

import { useState, useEffect } from 'react'
import { supabase } from '@/lib/supabase-client'

export interface Reward {
  id: string
  title: string
  description: string
  pointsRequired: number
  type: 'bonus' | 'badge' | 'feature'
  isUnlocked: boolean
  isRedeemed: boolean
  rewardValue?: number
}

export interface Achievement {
  id: string
  title: string
  description: string
  icon: string
  pointsAwarded: number
  isUnlocked: boolean
  unlockedAt?: string
}

export interface RewardsData {
  totalPoints: number
  currentLevel: string
  nextLevelPoints: number
  levelProgress: number
  monthlyPoints: number
  referralCount: number
  referralEarnings: number
  achievements: Achievement[]
  availableRewards: Reward[]
  redeemedRewards: Reward[]
}

export function useRewards(userId?: string) {
  const [rewardsData, setRewardsData] = useState<RewardsData>({
    totalPoints: 0,
    currentLevel: 'Bronze',
    nextLevelPoints: 100,
    levelProgress: 0,
    monthlyPoints: 0,
    referralCount: 0,
    referralEarnings: 0,
    achievements: [],
    availableRewards: [],
    redeemedRewards: []
  })
  const [loading, setLoading] = useState(true)
  const [error, setError] = useState<string | null>(null)

  useEffect(() => {
    const fetchRewardsData = async () => {
      if (!userId) {
        setLoading(false)
        return
      }

      try {
        setLoading(true)
        setError(null)

        // Fetch user profile for points and referral data
        const { data: profile, error: profileError } = await supabase
          .from('profiles')
          .select('total_points, referral_code')
          .eq('id', userId)
          .single()

        if (profileError) throw profileError

        // Fetch user's bets to calculate monthly points and achievements
        const thirtyDaysAgo = new Date()
        thirtyDaysAgo.setDate(thirtyDaysAgo.getDate() - 30)

        const { data: recentBets, error: betsError } = await supabase
          .from('bets')
          .select('status, amount, created_at, potential_payout')
          .eq('user_id', userId)
          .gte('created_at', thirtyDaysAgo.toISOString())

        if (betsError) throw betsError

        // Calculate monthly points from recent activity
        const monthlyPoints = calculateMonthlyPoints(recentBets || [])

        // Calculate level and progress
        const totalPoints = profile.total_points || 0
        const { currentLevel, nextLevelPoints, levelProgress } = calculateLevel(totalPoints)

        // Generate achievements based on user activity
        const achievements = generateAchievements(recentBets || [], totalPoints)

        // Generate available rewards
        const availableRewards = generateAvailableRewards(totalPoints)

        // Fetch referral data (simplified - you might want to create a referrals table)
        const { data: referrals, error: referralsError } = await supabase
          .from('profiles')
          .select('id')
          .eq('referred_by', userId)

        const referralCount = referrals?.length || 0
        const referralEarnings = referralCount * 0.10 // $0.10 per referral

        setRewardsData({
          totalPoints,
          currentLevel,
          nextLevelPoints,
          levelProgress,
          monthlyPoints,
          referralCount,
          referralEarnings,
          achievements,
          availableRewards,
          redeemedRewards: [] // TODO: Implement redeemed rewards tracking
        })

      } catch (err) {
        setError(err instanceof Error ? err.message : 'Failed to fetch rewards data')
        console.error('Error fetching rewards data:', err)
      } finally {
        setLoading(false)
      }
    }

    fetchRewardsData()
  }, [userId])

  const redeemReward = async (rewardId: string) => {
    try {
      setError(null)

      // Find the reward
      const reward = rewardsData.availableRewards.find(r => r.id === rewardId)
      if (!reward) {
        throw new Error('Reward not found')
      }

      // Check if user has enough points
      if (rewardsData.totalPoints < reward.pointsRequired) {
        throw new Error('Insufficient points')
      }

      // Update user's points in database
      const newPoints = rewardsData.totalPoints - reward.pointsRequired
      const { error: updateError } = await supabase
        .from('profiles')
        .update({ total_points: newPoints })
        .eq('id', userId)

      if (updateError) throw updateError

      // If it's a bonus reward, add to user's balance
      if (reward.type === 'bonus' && reward.rewardValue) {
        const { data: currentProfile } = await supabase
          .from('profiles')
          .select('balance')
          .eq('id', userId)
          .single()

        if (currentProfile) {
          const newBalance = (currentProfile.balance || 0) + reward.rewardValue
          await supabase
            .from('profiles')
            .update({ balance: newBalance })
            .eq('id', userId)
        }
      }

      // Update local state
      setRewardsData(prev => ({
        ...prev,
        totalPoints: newPoints,
        availableRewards: prev.availableRewards.filter(r => r.id !== rewardId),
        redeemedRewards: [...prev.redeemedRewards, { ...reward, isRedeemed: true }]
      }))

      return { success: true }
    } catch (err) {
      const errorMessage = err instanceof Error ? err.message : 'Failed to redeem reward'
      setError(errorMessage)
      return { success: false, error: errorMessage }
    }
  }

  return { rewardsData, loading, error, redeemReward }
}

// Helper functions
function calculateMonthlyPoints(bets: any[]): number {
  return bets.reduce((points, bet) => {
    if (bet.status === 'won') {
      return points + Math.floor((bet.potential_payout || bet.amount * 2) / 10)
    } else if (bet.status === 'matched') {
      return points + Math.floor(bet.amount / 20)
    }
    return points
  }, 0)
}

function calculateLevel(totalPoints: number): { currentLevel: string; nextLevelPoints: number; levelProgress: number } {
  const levels = [
    { name: 'Bronze', min: 0, max: 99 },
    { name: 'Silver', min: 100, max: 499 },
    { name: 'Gold', min: 500, max: 999 },
    { name: 'Platinum', min: 1000, max: 2499 },
    { name: 'Diamond', min: 2500, max: 4999 },
    { name: 'Master', min: 5000, max: 9999 },
    { name: 'Legend', min: 10000, max: Infinity }
  ]

  // Find the highest level the user qualifies for
  let currentLevel = levels[0]
  for (const level of levels) {
    if (totalPoints >= level.min) {
      currentLevel = level
    } else {
      break
    }
  }

  const nextLevel = levels[levels.indexOf(currentLevel) + 1] || levels[levels.length - 1]
  
  // Calculate progress towards next level
  let levelProgress = 0
  if (nextLevel && nextLevel.min !== Infinity) {
    const pointsInCurrentLevel = totalPoints - currentLevel.min
    const pointsNeededForNextLevel = nextLevel.min - currentLevel.min
    levelProgress = (pointsInCurrentLevel / pointsNeededForNextLevel) * 100
  } else {
    // If at max level (Legend), show 100% progress
    levelProgress = 100
  }

  return {
    currentLevel: currentLevel.name,
    nextLevelPoints: nextLevel.min,
    levelProgress: Math.max(0, Math.min(levelProgress, 100))
  }
}

function generateAchievements(bets: any[], totalPoints: number): Achievement[] {
  const achievements: Achievement[] = []

  // First bet achievement
  if (bets.length > 0) {
    achievements.push({
      id: 'first-bet',
      title: 'First Bet',
      description: 'Placed your first bet',
      icon: '🎯',
      pointsAwarded: 10,
      isUnlocked: true,
      unlockedAt: bets[0].created_at
    })
  }

  // First win achievement
  const wonBets = bets.filter(bet => bet.status === 'won')
  if (wonBets.length > 0) {
    achievements.push({
      id: 'first-win',
      title: 'First Win',
      description: 'Won your first bet',
      icon: '🏆',
      pointsAwarded: 50,
      isUnlocked: true,
      unlockedAt: wonBets[0].created_at
    })
  }

  // Streak achievements
  if (wonBets.length >= 3) {
    achievements.push({
      id: 'winning-streak',
      title: 'Winning Streak',
      description: 'Won 3 bets in a row',
      icon: '🔥',
      pointsAwarded: 100,
      isUnlocked: true,
      unlockedAt: wonBets[2].created_at
    })
  }

  // High roller achievement
  const totalBetAmount = bets.reduce((sum, bet) => sum + bet.amount, 0)
  if (totalBetAmount >= 1000) {
    achievements.push({
      id: 'high-roller',
      title: 'High Roller',
      description: 'Bet over $1000 total',
      icon: '💰',
      pointsAwarded: 200,
      isUnlocked: true,
      unlockedAt: new Date().toISOString()
    })
  }

  // Points milestones
  if (totalPoints >= 100) {
    achievements.push({
      id: 'century-club',
      title: 'Century Club',
      description: 'Earned 100 points',
      icon: '💯',
      pointsAwarded: 0,
      isUnlocked: true,
      unlockedAt: new Date().toISOString()
    })
  }

  return achievements
}

function generateAvailableRewards(totalPoints: number): Reward[] {
  const rewards: Reward[] = [
    {
      id: 'bonus-5',
      title: '$5 Bonus',
      description: 'Redeem 500 points for $5 account credit',
      pointsRequired: 500,
      type: 'bonus',
      rewardValue: 5,
      isUnlocked: totalPoints >= 500,
      isRedeemed: false
    },
    {
      id: 'bonus-10',
      title: '$10 Bonus',
      description: 'Redeem 900 points for $10 account credit',
      pointsRequired: 900,
      type: 'bonus',
      rewardValue: 10,
      isUnlocked: totalPoints >= 900,
      isRedeemed: false
    },
    {
      id: 'bonus-25',
      title: '$25 Bonus',
      description: 'Redeem 2000 points for $25 account credit',
      pointsRequired: 2000,
      type: 'bonus',
      rewardValue: 25,
      isUnlocked: totalPoints >= 2000,
      isRedeemed: false
    },
    {
      id: 'premium-badge',
      title: 'Premium Badge',
      description: 'Unlock premium features and golden badge',
      pointsRequired: 1000,
      type: 'badge',
      isUnlocked: totalPoints >= 1000,
      isRedeemed: false
    },
    {
      id: 'vip-status',
      title: 'VIP Status',
      description: 'Get VIP support and exclusive markets',
      pointsRequired: 5000,
      type: 'feature',
      isUnlocked: totalPoints >= 5000,
      isRedeemed: false
    }
  ]

  return rewards.filter(reward => reward.isUnlocked && !reward.isRedeemed)
}
