import { useState, useEffect } from 'react'
import { supabase } from '@/lib/supabase-client'

export interface UserPreferences {
  id: string
  user_id: string
  email_notifications: boolean
  email_market_updates: boolean
  email_bet_results: boolean
  email_security_alerts: boolean
  push_notifications: boolean
  theme: 'light' | 'dark' | 'system'
  language: string
  timezone: string
  public_profile: boolean
  show_betting_activity: boolean
  two_factor_enabled: boolean
  two_factor_method: 'sms' | 'authenticator'
  created_at: string
  updated_at: string
}

export function useUserPreferences(userId?: string) {
  const [preferences, setPreferences] = useState<UserPreferences | null>(null)
  const [loading, setLoading] = useState(true)
  const [error, setError] = useState<string | null>(null)

  useEffect(() => {
    if (!userId) {
      setLoading(false)
      return
    }

    fetchPreferences()
  }, [userId])

  const fetchPreferences = async () => {
    try {
      setLoading(true)
      setError(null)

      const { data, error: fetchError } = await supabase
        .rpc('get_user_preferences', { user_uuid: userId })

      if (fetchError) {
        throw fetchError
      }

      setPreferences(data)
    } catch (err) {
      console.error('Error fetching user preferences:', err)
      setError(err instanceof Error ? err.message : 'Failed to fetch preferences')
    } finally {
      setLoading(false)
    }
  }

  const updatePreferences = async (updates: Partial<UserPreferences>) => {
    try {
      setError(null)

      const { data, error: updateError } = await supabase
        .rpc('update_user_preferences', {
          user_uuid: userId,
          prefs_data: updates
        })

      if (updateError) {
        throw updateError
      }

      setPreferences(data)
      return data
    } catch (err) {
      console.error('Error updating user preferences:', err)
      setError(err instanceof Error ? err.message : 'Failed to update preferences')
      throw err
    }
  }

  const resetToDefaults = async () => {
    const defaultPreferences = {
      email_notifications: true,
      email_market_updates: true,
      email_bet_results: true,
      email_security_alerts: true,
      push_notifications: false,
      theme: 'light' as const,
      language: 'en',
      timezone: 'UTC',
      public_profile: false,
      show_betting_activity: true,
      two_factor_enabled: false,
      two_factor_method: 'sms' as const
    }

    return await updatePreferences(defaultPreferences)
  }

  return {
    preferences,
    loading,
    error,
    updatePreferences,
    resetToDefaults,
    refetch: fetchPreferences
  }
}
