import { useState, useEffect, useCallback, useMemo } from 'react'
import { supabase } from '@/lib/supabase-client'

export interface UserPreferences {
  id?: string
  user_id: string
  email_notifications: boolean
  email_market_updates: boolean
  email_bet_results: boolean
  email_security_alerts: boolean
  push_notifications: boolean
  theme: 'light' | 'dark' | 'system'
  language: string
  timezone: string
  public_profile: boolean
  show_betting_activity: boolean
  two_factor_enabled: boolean
  two_factor_method: 'sms' | 'authenticator'
  created_at?: string
  updated_at?: string
}

const DEFAULT_PREFERENCES: Omit<UserPreferences, 'user_id'> = {
  email_notifications: true,
  email_market_updates: true,
  email_bet_results: true,
  email_security_alerts: true,
  push_notifications: false,
  theme: 'light',
  language: 'en',
  timezone: 'UTC',
  public_profile: false,
  show_betting_activity: true,
  two_factor_enabled: false,
  two_factor_method: 'sms'
}

export function useUserPreferencesSimple(userId?: string) {
  const [preferences, setPreferences] = useState<UserPreferences | null>(null)
  const [loading, setLoading] = useState(true)
  const [error, setError] = useState<string | null>(null)

  const fetchPreferences = useCallback(async () => {
    if (!userId) {
      setLoading(false)
      return
    }

    try {
      setLoading(true)
      setError(null)

      // Try to fetch from user_preferences table first
      const { data: prefsData, error: prefsError } = await supabase
        .from('user_preferences')
        .select('*')
        .eq('user_id', userId)
        .single()

      if (prefsError && prefsError.code !== 'PGRST116') { // PGRST116 = no rows returned
        throw prefsError
      }

      if (prefsData) {
        setPreferences(prefsData as UserPreferences)
      } else {
        // Create default preferences
        const defaultPrefs: UserPreferences = {
          ...DEFAULT_PREFERENCES,
          user_id: userId
        }
        setPreferences(defaultPrefs)
      }
    } catch (err) {
      console.error('Error fetching preferences:', err)
      setError(err instanceof Error ? err.message : 'Failed to fetch preferences')
      
      // Set default preferences on error
      const defaultPrefs: UserPreferences = {
        ...DEFAULT_PREFERENCES,
        user_id: userId
      }
      setPreferences(defaultPrefs)
    } finally {
      setLoading(false)
    }
  }, [userId])

  useEffect(() => {
    fetchPreferences()
  }, [fetchPreferences])

  const updatePreferences = useCallback(async (updates: Partial<UserPreferences>) => {
    if (!userId) return

    try {
      setError(null)

      const updatedPrefs = { ...preferences, ...updates }

      // Try to update existing preferences
      const { data, error: updateError } = await supabase
        .from('user_preferences')
        .upsert({
          user_id: userId,
          ...updatedPrefs
        })
        .select()
        .single()

      if (updateError) {
        throw updateError
      }

      setPreferences(data as UserPreferences)
      return data
    } catch (err) {
      console.error('Error updating preferences:', err)
      setError(err instanceof Error ? err.message : 'Failed to update preferences')
      
      // Update local state even if database update fails
      setPreferences(prev => prev ? { ...prev, ...updates } : null)
      throw err
    }
  }, [userId, preferences])

  const resetToDefaults = useCallback(async () => {
    const defaultPrefs: UserPreferences = {
      ...DEFAULT_PREFERENCES,
      user_id: userId!
    }
    return await updatePreferences(defaultPrefs)
  }, [userId, updatePreferences])

  // Memoize the return value
  return useMemo(() => ({
    preferences,
    loading,
    error,
    updatePreferences,
    resetToDefaults,
    refetch: fetchPreferences
  }), [preferences, loading, error, updatePreferences, resetToDefaults, fetchPreferences])
}
