'use client'

import { useState, useEffect } from 'react'
import { supabase } from '@/lib/supabase-client'

interface UserStats {
  rank: number
  points: number
  winRate: number
  profit: number
  totalBets: number
}

interface PlatformStats {
  totalChampions: number
  totalProfits: number
  averageWinRate: number
}

export function useUserStats(userId?: string, timePeriod: 'all' | 'month' | 'week' = 'month') {
  const [userStats, setUserStats] = useState<UserStats>({
    rank: 0,
    points: 0,
    winRate: 0,
    profit: 0,
    totalBets: 0
  })
  const [platformStats, setPlatformStats] = useState<PlatformStats>({
    totalChampions: 0,
    totalProfits: 0,
    averageWinRate: 0
  })
  const [loading, setLoading] = useState(true)
  const [error, setError] = useState<string | null>(null)

  useEffect(() => {
    const fetchStats = async () => {
      if (!userId) {
        setLoading(false)
        return
      }

      try {
        setLoading(true)
        setError(null)

        // Fetch user's profile and bets
        const [profileResult, betsResult, allProfilesResult] = await Promise.all([
          supabase
            .from('profiles')
            .select('total_points')
            .eq('id', userId)
            .single(),
          
          (() => {
            let query = supabase
              .from('bets')
              .select('status, amount, potential_payout')
              .eq('user_id', userId)
            
            // Apply date filter if not 'all' time period
            if (timePeriod !== 'all') {
              const now = new Date()
              if (timePeriod === 'week') {
                const weekAgo = new Date(now.getTime() - 7 * 24 * 60 * 60 * 1000)
                query = query.gte('created_at', weekAgo.toISOString())
              } else if (timePeriod === 'month') {
                const monthAgo = new Date(now.getTime() - 30 * 24 * 60 * 60 * 1000)
                query = query.gte('created_at', monthAgo.toISOString())
              }
            }
            
            return query
          })(),
          
          supabase
            .from('profiles')
            .select('id, total_points')
            .eq('account_status', 'active')
            .order('total_points', { ascending: false })
        ])

        if (profileResult.error) throw profileResult.error
        if (betsResult.error) throw betsResult.error
        if (allProfilesResult.error) throw allProfilesResult.error

        const profile = profileResult.data
        const userBets = betsResult.data || []
        const allProfiles = allProfilesResult.data || []

        // Calculate user's rank
        const userRank = allProfiles.findIndex(p => p.id === userId) + 1

        // Calculate user's win rate
        const wonBets = userBets.filter(bet => bet.status === 'won').length
        const totalCompletedBets = userBets.filter(bet => bet.status === 'won' || bet.status === 'lost').length
        const winRate = totalCompletedBets > 0 ? Math.round((wonBets / totalCompletedBets) * 100) : 0

        // Calculate user's profit
        const profit = userBets.reduce((sum, bet) => {
          if (bet.status === 'won') {
            return sum + (bet.potential_payout || 0)
          } else if (bet.status === 'lost') {
            return sum - (bet.amount || 0)
          }
          return sum
        }, 0)

        // Calculate platform stats
        const totalChampions = allProfiles.filter(p => p.total_points > 0).length
        const totalProfits = allProfiles.reduce((sum, profile) => sum + (profile.total_points || 0), 0)

        // Calculate average win rate across all users with date filter
        let allBetsQuery = supabase
          .from('bets')
          .select('status')
          .in('status', ['won', 'lost'])
        
        // Apply date filter if not 'all' time period
        if (timePeriod !== 'all') {
          const now = new Date()
          if (timePeriod === 'week') {
            const weekAgo = new Date(now.getTime() - 7 * 24 * 60 * 60 * 1000)
            allBetsQuery = allBetsQuery.gte('created_at', weekAgo.toISOString())
          } else if (timePeriod === 'month') {
            const monthAgo = new Date(now.getTime() - 30 * 24 * 60 * 60 * 1000)
            allBetsQuery = allBetsQuery.gte('created_at', monthAgo.toISOString())
          }
        }
        
        const allBetsResult = await allBetsQuery

        if (!allBetsResult.error && allBetsResult.data) {
          const allBets = allBetsResult.data
          const totalWonBets = allBets.filter(bet => bet.status === 'won').length
          const totalCompletedBets = allBets.length
          const averageWinRate = totalCompletedBets > 0 ? Math.round((totalWonBets / totalCompletedBets) * 100) : 0

          setPlatformStats({
            totalChampions,
            totalProfits,
            averageWinRate
          })
        }

        setUserStats({
          rank: userRank,
          points: profile?.total_points || 0,
          winRate,
          profit: Math.round(profit),
          totalBets: userBets.length
        })

      } catch (err) {
        setError(err instanceof Error ? err.message : 'Failed to fetch user stats')
        console.error('Error fetching user stats:', err)
      } finally {
        setLoading(false)
      }
    }

    fetchStats()
  }, [userId, timePeriod])

  return { userStats, platformStats, loading, error }
}
