'use client'

import { useState, useEffect } from 'react'
import { supabase } from '@/lib/supabase-client'
import { WalletTransaction } from '@/lib/types'

interface WalletData {
  balance: number
  transactions: WalletTransaction[]
}

export function useWallet(userId: string) {
  const [data, setData] = useState<WalletData>({
    balance: 0,
    transactions: []
  })
  const [loading, setLoading] = useState(true)
  const [error, setError] = useState<string | null>(null)

  useEffect(() => {
    if (!userId) {
      setLoading(false)
      return
    }

    async function fetchWalletData() {
      try {
        setLoading(true)
        setError(null)

        // Fetch user balance from profile
        const { data: profile, error: profileError } = await supabase
          .from('profiles')
          .select('balance')
          .eq('id', userId)
          .single()

        if (profileError) {
          console.error('Error fetching profile balance:', profileError)
        }

        // Fetch all transactions from unified table (including all types: deposits, withdrawals, bets, wins, seeds, liquidations, etc.)
        const { data: transactions, error: transactionsError } = await supabase
          .from('unified_transactions')
          .select(`
            *,
            markets (
              id,
              title
            )
          `)
          .eq('user_id', userId)
          .order('created_at', { ascending: false })
          .limit(50)

        if (transactionsError) {
          console.error('Error fetching transactions:', transactionsError)
        }

        setData({
          balance: profile?.balance || 0,
          transactions: transactions || []
        })
      } catch (err) {
        console.error('Error in fetchWalletData:', err)
        setError('Failed to fetch wallet data')
      } finally {
        setLoading(false)
      }
    }

    fetchWalletData()

    // Set up real-time subscriptions
    const profileSubscription = supabase
      .channel('profile-balance-changes')
      .on(
        'postgres_changes',
        {
          event: 'UPDATE',
          schema: 'public',
          table: 'profiles',
          filter: `id=eq.${userId}`
        },
        (payload) => {
          console.log('Profile balance updated:', payload)
          setData(prev => ({
            ...prev,
            balance: payload.new.balance || 0
          }))
        }
      )
      .subscribe()

    const transactionsSubscription = supabase
      .channel('wallet-transactions-changes')
      .on(
        'postgres_changes',
        {
          event: '*',
          schema: 'public',
          table: 'unified_transactions',
          filter: `user_id=eq.${userId}`
        },
        (payload) => {
          console.log('Transaction updated:', payload)
          // Refetch transactions to get the latest data
          fetchWalletData()
        }
      )
      .subscribe()

    // Cleanup subscriptions on unmount
    return () => {
      profileSubscription.unsubscribe()
      transactionsSubscription.unsubscribe()
    }
  }, [userId])

  return { ...data, loading, error }
}
