import { useState, useEffect, useCallback, useMemo } from 'react'
import { supabase } from '@/lib/supabase-client'
import { Profile, PaymentMethod, WalletTransaction, DailyLimits } from '@/lib/types'

interface WalletData {
  profile: Profile
  transactions: WalletTransaction[]
  paymentMethods: PaymentMethod[]
  dailyLimits: DailyLimits | null
}

export function useWalletOptimized(userId: string) {
  const [walletData, setWalletData] = useState<WalletData | null>(null)
  const [loading, setLoading] = useState(true)
  const [error, setError] = useState<string | null>(null)
  const [lastFetch, setLastFetch] = useState<number>(0)

  // Debounced fetch function
  const debouncedFetch = useCallback(
    (() => {
      let timeoutId: NodeJS.Timeout
      return (force = false) => {
        const now = Date.now()
        if (force || now - lastFetch > 3000) { // 3 second throttle
          clearTimeout(timeoutId)
          timeoutId = setTimeout(() => {
            fetchWalletData(force)
            setLastFetch(Date.now())
          }, 500) // 500ms debounce
        }
      }
    })(),
    [lastFetch]
  )

  const fetchWalletData = useCallback(async (isBackground = false) => {
    try {
      if (!isBackground) {
        setLoading(true)
        setError(null)
      }

      // Fetch data in parallel with optimized queries
      const [profileResult, transactionsResult, paymentMethodsResult] = await Promise.allSettled([
        // Profile with minimal data
        supabase
          .from('profiles')
          .select(`
            id,
            username,
            email,
            balance,
            kyc_verified,
            daily_deposit_limit,
            daily_withdrawal_limit,
            monthly_deposit_limit,
            monthly_withdrawal_limit,
            preferred_payment_method
          `)
          .eq('id', userId)
          .single(),

        // Transactions with pagination
        supabase
          .from('unified_transactions')
          .select(`
            id,
            type,
            amount,
            status,
            created_at,
            description,
            market_id,
            markets!inner(id, title)
          `)
          .eq('user_id', userId)
          .order('created_at', { ascending: false })
          .limit(15), // Reduced limit

        // Payment methods
        supabase
          .from('payment_methods')
          .select('*')
          .eq('user_id', userId)
          .eq('is_active', true)
      ])

      // Process results safely
      const profile = profileResult.status === 'fulfilled' && profileResult.value.data 
        ? profileResult.value.data as Profile
        : null

      const transactions = transactionsResult.status === 'fulfilled' && transactionsResult.value.data 
        ? transactionsResult.value.data as WalletTransaction[]
        : []

      const paymentMethods = paymentMethodsResult.status === 'fulfilled' && paymentMethodsResult.value.data 
        ? paymentMethodsResult.value.data as PaymentMethod[]
        : []

      // Calculate daily limits
      const dailyLimits: DailyLimits | null = profile ? {
        daily_deposit_limit: profile.daily_deposit_limit || 5000,
        daily_withdrawal_limit: profile.daily_withdrawal_limit || 5000,
        monthly_deposit_limit: profile.monthly_deposit_limit || 50000,
        monthly_withdrawal_limit: profile.monthly_withdrawal_limit || 50000
      } : null

      if (profile) {
        setWalletData({
          profile,
          transactions,
          paymentMethods,
          dailyLimits
        })
      }

    } catch (err) {
      console.error('Error fetching wallet data:', err)
      if (!isBackground) {
        setError(err instanceof Error ? err.message : 'Failed to fetch wallet data')
      }
    } finally {
      if (!isBackground) {
        setLoading(false)
      }
    }
  }, [userId])

  useEffect(() => {
    fetchWalletData()

    // Set up optimized real-time subscriptions
    let subscriptions: any[] = []

    // Profile subscription - only for balance updates
    const profileSubscription = supabase
      .channel(`wallet-profile-${userId}`)
      .on(
        'postgres_changes',
        {
          event: 'UPDATE',
          schema: 'public',
          table: 'profiles',
          filter: `id=eq.${userId}`
        },
        (payload) => {
          setWalletData(prev => prev ? {
            ...prev,
            profile: payload.new as Profile
          } : null)
        }
      )
      .subscribe()

    // Transactions subscription - debounced refetch
    const transactionsSubscription = supabase
      .channel(`wallet-transactions-${userId}`)
      .on(
        'postgres_changes',
        {
          event: 'INSERT',
          schema: 'public',
          table: 'unified_transactions',
          filter: `user_id=eq.${userId}`
        },
        () => {
          debouncedFetch()
        }
      )
      .subscribe()

    subscriptions = [profileSubscription, transactionsSubscription]

    // Cleanup subscriptions on unmount
    return () => {
      subscriptions.forEach(sub => sub.unsubscribe())
    }
  }, [userId, fetchWalletData, debouncedFetch])

  // Memoize the return value
  return useMemo(() => ({
    profile: walletData?.profile || null,
    transactions: walletData?.transactions || [],
    paymentMethods: walletData?.paymentMethods || [],
    dailyLimits: walletData?.dailyLimits || null,
    loading,
    error,
    refetch: () => fetchWalletData(true)
  }), [walletData, loading, error, fetchWalletData])
}
