import { useState, useEffect, useCallback } from 'react'
import { supabase } from '@/lib/supabase-client'
import { useToast } from '@/components/ui/Toast'

interface WatchlistItem {
  id: string
  market_id: string
  created_at: string
  markets: {
    id: string
    title: string
    description: string
    category: string
    status: string
    yes_price: number
    no_price: number
    total_volume: number
    end_date: string
    created_at: string
    creator_id: string
    creator: {
      email: string
    }
  }
}

export function useWatchlist(userId?: string) {
  const [watchlistItems, setWatchlistItems] = useState<WatchlistItem[]>([])
  const [bookmarkedMarkets, setBookmarkedMarkets] = useState<string[]>([])
  const [loading, setLoading] = useState(false)
  const [error, setError] = useState<string | null>(null)
  const { addToast } = useToast()

  // Fetch user's watchlist
  const fetchWatchlist = useCallback(async () => {
    if (!userId) {
      setWatchlistItems([])
      setBookmarkedMarkets([])
      return
    }

    try {
      setLoading(true)
      setError(null)

      const { data, error } = await supabase
        .from('watchlist')
        .select(`
          id,
          market_id,
          created_at,
          markets (
            id,
            title,
            description,
            category,
            status,
            yes_price,
            no_price,
            total_volume,
            end_date,
            created_at,
            creator_id,
            creator:profiles(email)
          )
        `)
        .eq('user_id', userId)
        .order('created_at', { ascending: false })

      if (error) {
        console.warn('Watchlist not available:', error.message)
        // Don't set error for watchlist, it's optional
        setWatchlistItems([])
        setBookmarkedMarkets([])
        return
      }

      const items = data || []
      setWatchlistItems(items)
      setBookmarkedMarkets(items.map(item => item.market_id))
    } catch (err) {
      console.error('Error fetching watchlist:', err)
      setError('Failed to load watchlist')
    } finally {
      setLoading(false)
    }
  }, [userId])

  // Add to watchlist
  const addToWatchlist = useCallback(async (marketId: string) => {
    if (!userId) return false

    try {
      const { error } = await supabase
        .from('watchlist')
        .insert({
          user_id: userId,
          market_id: marketId,
          created_at: new Date().toISOString()
        })

      if (error) {
        console.error('Error adding to watchlist:', error)
        addToast({
          type: 'error',
          title: 'Error',
          message: 'Failed to add to watchlist'
        })
        return false
      }

      setBookmarkedMarkets(prev => [...prev, marketId])
      addToast({
        type: 'success',
        title: 'Added',
        message: 'Market added to watchlist'
      })
      return true
    } catch (err) {
      console.error('Error adding to watchlist:', err)
      addToast({
        type: 'error',
        title: 'Error',
        message: 'Failed to add to watchlist'
      })
      return false
    }
  }, [userId, addToast])

  // Remove from watchlist
  const removeFromWatchlist = useCallback(async (marketId: string) => {
    if (!userId) return false

    try {
      const { error } = await supabase
        .from('watchlist')
        .delete()
        .eq('user_id', userId)
        .eq('market_id', marketId)

      if (error) {
        console.error('Error removing from watchlist:', error)
        addToast({
          type: 'error',
          title: 'Error',
          message: 'Failed to remove from watchlist'
        })
        return false
      }

      setWatchlistItems(prev => prev.filter(item => item.market_id !== marketId))
      setBookmarkedMarkets(prev => prev.filter(id => id !== marketId))
      addToast({
        type: 'success',
        title: 'Removed',
        message: 'Market removed from watchlist'
      })
      return true
    } catch (err) {
      console.error('Error removing from watchlist:', err)
      addToast({
        type: 'error',
        title: 'Error',
        message: 'Failed to remove from watchlist'
      })
      return false
    }
  }, [userId, addToast])

  // Toggle bookmark
  const toggleBookmark = useCallback(async (marketId: string) => {
    if (!userId) return false

    const isBookmarked = bookmarkedMarkets.includes(marketId)
    
    if (isBookmarked) {
      return await removeFromWatchlist(marketId)
    } else {
      return await addToWatchlist(marketId)
    }
  }, [userId, bookmarkedMarkets, addToWatchlist, removeFromWatchlist])

  // Check if market is bookmarked
  const isBookmarked = useCallback((marketId: string) => {
    return bookmarkedMarkets.includes(marketId)
  }, [bookmarkedMarkets])

  // Fetch watchlist on mount and when userId changes
  useEffect(() => {
    fetchWatchlist()
  }, [fetchWatchlist])

  return {
    watchlistItems,
    bookmarkedMarkets,
    loading,
    error,
    addToWatchlist,
    removeFromWatchlist,
    toggleBookmark,
    isBookmarked,
    refreshWatchlist: fetchWatchlist
  }
}

