interface LogEntry {
  timestamp: string
  level: 'info' | 'warn' | 'error' | 'debug'
  message: string
  data?: any
  userId?: string
  sessionId?: string
}

class Logger {
  private logs: LogEntry[] = []
  private maxLogs = 1000

  log(level: LogEntry['level'], message: string, data?: any, userId?: string) {
    const entry: LogEntry = {
      timestamp: new Date().toISOString(),
      level,
      message,
      data,
      userId,
      sessionId: this.getSessionId(),
    }

    this.logs.push(entry)
    
    // Keep only the last maxLogs entries
    if (this.logs.length > this.maxLogs) {
      this.logs = this.logs.slice(-this.maxLogs)
    }

    // Console logging
    const consoleMethod = level === 'error' ? 'error' : level === 'warn' ? 'warn' : 'log'
    console[consoleMethod](`[${level.toUpperCase()}] ${message}`, data || '')

    // Send to external service in production
    if (process.env.NODE_ENV === 'production') {
      this.sendToExternalService(entry)
    }
  }

  info(message: string, data?: any, userId?: string) {
    this.log('info', message, data, userId)
  }

  warn(message: string, data?: any, userId?: string) {
    this.log('warn', message, data, userId)
  }

  error(message: string, data?: any, userId?: string) {
    this.log('error', message, data, userId)
  }

  debug(message: string, data?: any, userId?: string) {
    this.log('debug', message, data, userId)
  }

  getLogs(): LogEntry[] {
    return [...this.logs]
  }

  getLogsByLevel(level: LogEntry['level']): LogEntry[] {
    return this.logs.filter(log => log.level === level)
  }

  getLogsByUser(userId: string): LogEntry[] {
    return this.logs.filter(log => log.userId === userId)
  }

  clearLogs() {
    this.logs = []
  }

  private getSessionId(): string {
    if (typeof window !== 'undefined') {
      return sessionStorage.getItem('sessionId') || 'unknown'
    }
    return 'server'
  }

  private async sendToExternalService(entry: LogEntry) {
    try {
      // In production, you might want to send logs to an external service
      // like Sentry, LogRocket, or your own logging service
      await fetch('/api/logs', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify(entry),
      })
    } catch (error) {
      console.error('Failed to send log to external service:', error)
    }
  }
}

export const logger = new Logger()
