// Unified Payment Service for BetFusion Zim
// Handles PayNow Zim and Pesapal payment methods

import { payNowZimService, PayNowZimPaymentRequest, PayNowZimPaymentResponse } from './paynow-zim'
import { pesapalService, PesapalPaymentRequest, PesapalPaymentResponse } from './pesapal'

export type PaymentProvider = 'paynow_zim' | 'pesapal'

export interface UnifiedPaymentRequest {
  provider: PaymentProvider
  amount: number
  currency: 'USD' // USD-only implementation
  description: string
  reference: string
  customerEmail: string
  customerPhone?: string
  customerFirstName?: string
  customerLastName?: string
  returnUrl: string
  cancelUrl: string
}

export interface UnifiedPaymentResponse {
  success: boolean
  provider: PaymentProvider
  paymentUrl?: string
  transactionId?: string
  orderTrackingId?: string
  error?: string
  message?: string
}

export interface PaymentStatusResponse {
  success: boolean
  status: 'pending' | 'completed' | 'failed' | 'cancelled'
  provider: PaymentProvider
  transactionId?: string
  orderTrackingId?: string
  error?: string
}

export class UnifiedPaymentService {
  /**
   * Create a payment request with the specified provider
   */
  async createPayment(request: UnifiedPaymentRequest): Promise<UnifiedPaymentResponse> {
    try {
      switch (request.provider) {
        case 'paynow_zim':
          return await this.createPayNowZimPayment(request)
        case 'pesapal':
          return await this.createPesapalPayment(request)
        default:
          throw new Error(`Unsupported payment provider: ${request.provider}`)
      }
    } catch (error) {
      console.error('Payment creation failed:', error)
      return {
        success: false,
        provider: request.provider,
        error: error instanceof Error ? error.message : 'Payment creation failed'
      }
    }
  }

  /**
   * Create PayNow Zim payment
   */
  private async createPayNowZimPayment(request: UnifiedPaymentRequest): Promise<UnifiedPaymentResponse> {
    const payNowRequest: PayNowZimPaymentRequest = {
      amount: request.amount,
      currency: request.currency === 'ZWL' ? 'ZWL' : 'USD',
      customerEmail: request.customerEmail,
      customerPhone: request.customerPhone,
      description: request.description,
      reference: request.reference,
      returnUrl: request.returnUrl,
      cancelUrl: request.cancelUrl
    }

    const response = await payNowZimService.createPayment(payNowRequest)
    
    return {
      success: response.success,
      provider: 'paynow_zim',
      paymentUrl: response.paymentUrl,
      transactionId: response.transactionId,
      error: response.error,
      message: response.message
    }
  }

  /**
   * Create Pesapal payment
   */
  private async createPesapalPayment(request: UnifiedPaymentRequest): Promise<UnifiedPaymentResponse> {
    const pesapalRequest: PesapalPaymentRequest = {
      amount: request.amount,
      currency: request.currency,
      description: request.description,
      type: 'MERCHANT',
      reference: request.reference,
      firstName: request.customerFirstName || 'Customer',
      lastName: request.customerLastName || 'User',
      email: request.customerEmail,
      phoneNumber: request.customerPhone,
      callbackUrl: request.returnUrl,
      cancellationUrl: request.cancelUrl,
      notificationUrl: `${process.env.NEXT_PUBLIC_BASE_URL}/api/webhooks/pesapal`
    }

    const response = await pesapalService.createPayment(pesapalRequest)
    
    return {
      success: response.success,
      provider: 'pesapal',
      paymentUrl: response.redirectUrl,
      orderTrackingId: response.orderTrackingId,
      error: response.error,
      message: response.message
    }
  }

  /**
   * Get payment status
   */
  async getPaymentStatus(provider: PaymentProvider, transactionId: string): Promise<PaymentStatusResponse> {
    try {
      switch (provider) {
        case 'paynow_zim':
          const payNowResponse = await payNowZimService.getPaymentStatus(transactionId)
          return {
            success: payNowResponse.success,
            status: 'completed', // In real implementation, parse from response
            provider: 'paynow_zim',
            transactionId: transactionId,
            error: payNowResponse.error
          }
        case 'pesapal':
          const pesapalResponse = await pesapalService.getPaymentStatus(transactionId)
          return {
            success: pesapalResponse.success,
            status: 'completed', // In real implementation, parse from response
            provider: 'pesapal',
            orderTrackingId: transactionId,
            error: pesapalResponse.error
          }
        default:
          throw new Error(`Unsupported payment provider: ${provider}`)
      }
    } catch (error) {
      return {
        success: false,
        status: 'failed',
        provider,
        error: error instanceof Error ? error.message : 'Status check failed'
      }
    }
  }

  /**
   * Get supported payment methods for a given currency
   * USD-only implementation
   */
  getSupportedPaymentMethods(currency: string): PaymentProvider[] {
    const methods: PaymentProvider[] = []
    
    // Only support USD currency
    if (currency === 'USD') {
      methods.push('paynow_zim')
      methods.push('pesapal')
    }
    
    return methods
  }

  /**
   * Get payment method display information
   */
  getPaymentMethodInfo(provider: PaymentProvider): {
    name: string
    description: string
    icon: string
    supportedCurrencies: string[]
    processingTime: string
  } {
    switch (provider) {
      case 'paynow_zim':
        return {
          name: 'PayNow Zim',
          description: 'Mobile money and digital payments for Zimbabwe (USD only)',
          icon: '🇿🇼',
          supportedCurrencies: ['USD'],
          processingTime: 'Instant'
        }
      case 'pesapal':
        return {
          name: 'Pesapal',
          description: 'Leading African payment gateway (USD only)',
          icon: '🌍',
          supportedCurrencies: ['USD'],
          processingTime: '1-3 minutes'
        }
      default:
        throw new Error(`Unknown payment provider: ${provider}`)
    }
  }
}

// Export service instance
export const unifiedPaymentService = new UnifiedPaymentService()

// Export types and services
export * from './paynow-zim'
export * from './pesapal'
