// PayNow Zim Payment Integration
// PayNow Zim is a popular mobile money and payment gateway in Zimbabwe

import { Paynow } from 'paynow'

export interface PayNowZimConfig {
  integrationId: string
  integrationKey: string
  baseUrl: string
  webhookSecret: string
}

export interface PayNowZimPaymentRequest {
  amount: number
  currency: 'USD' // USD-only implementation
  customerEmail: string
  customerPhone?: string
  customerFirstName?: string
  customerLastName?: string
  description: string
  reference: string
  returnUrl: string
  cancelUrl: string
}

export interface PayNowZimPaymentResponse {
  success: boolean
  paymentUrl?: string
  transactionId?: string
  error?: string
  message?: string
}

export interface PayNowZimWebhookPayload {
  transactionId: string
  reference: string
  status: 'pending' | 'completed' | 'failed' | 'cancelled'
  amount: number
  currency: string
  timestamp: string
  signature: string
}

export class PayNowZimService {
  private config: PayNowZimConfig
  private paynow: Paynow

  constructor(config: PayNowZimConfig) {
    this.config = config
    this.paynow = new Paynow(config.integrationId, config.integrationKey)
  }

  /**
   * Create a payment request with PayNow Zim
   */
  async createPayment(request: PayNowZimPaymentRequest): Promise<PayNowZimPaymentResponse> {
    try {
      // Set return and result URLs
      this.paynow.resultUrl = `${this.config.baseUrl}/api/webhooks/paynow-zim`
      this.paynow.returnUrl = request.returnUrl

      // Create a new payment
      const payment = this.paynow.createPayment(request.reference, request.customerEmail)

      // Add the payment item
      payment.add(request.description, request.amount)

      // Send the payment to PayNow
      const response = await this.paynow.send(payment)

      if (response.success) {
        return {
          success: true,
          paymentUrl: response.redirectUrl,
          transactionId: response.pollUrl, // Use pollUrl as transaction ID for tracking
          message: 'Payment request created successfully'
        }
      } else {
        return {
          success: false,
          error: response.error || 'Payment creation failed'
        }
      }
    } catch (error) {
      console.error('PayNow Zim payment creation failed:', error)
      return {
        success: false,
        error: error instanceof Error ? error.message : 'Payment creation failed'
      }
    }
  }

  /**
   * Check payment status using PayNow polling
   */
  async checkPaymentStatus(pollUrl: string): Promise<PayNowZimPaymentResponse> {
    try {
      const status = await this.paynow.pollTransaction(pollUrl)
      
      if (status.paid()) {
        return {
          success: true,
          message: 'Payment completed successfully'
        }
      } else {
        return {
          success: false,
          error: 'Payment not completed'
        }
      }
    } catch (error) {
      console.error('PayNow status check failed:', error)
      return {
        success: false,
        error: error instanceof Error ? error.message : 'Status check failed'
      }
    }
  }

  /**
   * Verify webhook signature from PayNow Zim
   */
  verifyWebhookSignature(payload: any, signature: string): boolean {
    // In a real implementation, you would verify the signature using HMAC
    // For now, we'll simulate verification
    return true
  }

  /**
   * Process webhook notification from PayNow Zim
   */
  async processWebhook(payload: PayNowZimWebhookPayload): Promise<{ success: boolean; error?: string }> {
    try {
      // Verify signature
      if (!this.verifyWebhookSignature(payload, payload.signature)) {
        return { success: false, error: 'Invalid signature' }
      }

      // Process the webhook based on status
      switch (payload.status) {
        case 'completed':
          // Update transaction status in database
          await this.updateTransactionStatus(payload.transactionId, 'completed')
          break
        case 'failed':
        case 'cancelled':
          await this.updateTransactionStatus(payload.transactionId, 'failed')
          break
        case 'pending':
          // Keep as pending, no action needed
          break
      }

      return { success: true }
    } catch (error) {
      console.error('PayNow Zim webhook processing failed:', error)
      return { 
        success: false, 
        error: error instanceof Error ? error.message : 'Webhook processing failed' 
      }
    }
  }

  /**
   * Get payment status from PayNow Zim
   */
  async getPaymentStatus(transactionId: string): Promise<PayNowZimPaymentResponse> {
    try {
      // In a real implementation, this would query PayNow Zim API
      const status = await this.simulateStatusCheck(transactionId)
      
      return {
        success: true,
        message: `Payment status: ${status}`
      }
    } catch (error) {
      return {
        success: false,
        error: error instanceof Error ? error.message : 'Status check failed'
      }
    }
  }

  /**
   * Simulate PayNow Zim API call (replace with actual API integration)
   */
  private async simulatePayNowZimApiCall(payload: any): Promise<any> {
    // Simulate API delay
    await new Promise(resolve => setTimeout(resolve, 1000))
    
    return {
      transaction_id: `pz_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`,
      payment_url: `https://paynow.zim/pay/${payload.reference}?amount=${payload.amount}&currency=${payload.currency}`,
      status: 'pending'
    }
  }

  /**
   * Simulate status check (replace with actual API call)
   */
  private async simulateStatusCheck(transactionId: string): Promise<string> {
    await new Promise(resolve => setTimeout(resolve, 500))
    return 'completed'
  }

  /**
   * Update transaction status in database
   */
  private async updateTransactionStatus(transactionId: string, status: string): Promise<void> {
    // In a real implementation, this would update the database
    console.log(`Updating transaction ${transactionId} status to ${status}`)
  }
}

// Default configuration (replace with actual credentials)
export const defaultPayNowZimConfig: PayNowZimConfig = {
  integrationId: process.env.PAYNOW_ZIM_INTEGRATION_ID || 'your_integration_id',
  integrationKey: process.env.PAYNOW_ZIM_INTEGRATION_KEY || 'your_integration_key',
  baseUrl: process.env.PAYNOW_ZIM_BASE_URL || 'https://api.paynow.co.zw',
  webhookSecret: process.env.PAYNOW_ZIM_WEBHOOK_SECRET || 'your_webhook_secret'
}

// Export service instance
export const payNowZimService = new PayNowZimService(defaultPayNowZimConfig)
