// Pesapal Payment Integration
// Pesapal is a leading payment gateway in Africa, supporting multiple payment methods

export interface PesapalConfig {
  consumerKey: string
  consumerSecret: string
  baseUrl: string
  webhookSecret: string
  environment: 'sandbox' | 'production'
}

export interface PesapalPaymentRequest {
  amount: number
  currency: 'USD' | 'KES' | 'UGX' | 'TZS' | 'ZAR' | 'GHS' | 'NGN'
  description: string
  type: 'MERCHANT'
  reference: string
  firstName: string
  lastName: string
  email: string
  phoneNumber?: string
  callbackUrl: string
  cancellationUrl: string
  notificationUrl: string
}

export interface PesapalPaymentResponse {
  success: boolean
  orderTrackingId?: string
  merchantReference?: string
  redirectUrl?: string
  error?: string
  message?: string
}

export interface PesapalWebhookPayload {
  orderTrackingId: string
  orderMerchantReference: string
  orderNotificationType: 'IPN' | 'PaymentConfirmation'
  orderStatus: 'COMPLETED' | 'FAILED' | 'PENDING'
  paymentMethod: string
  paymentAccount: string
  currency: string
  amount: number
  paymentStatus: 'COMPLETED' | 'FAILED' | 'PENDING'
  timestamp: string
  signature: string
}

export class PesapalService {
  private config: PesapalConfig
  private accessToken: string | null = null
  private tokenExpiry: number = 0

  constructor(config: PesapalConfig) {
    this.config = config
  }

  /**
   * Get access token from Pesapal
   */
  private async getAccessToken(): Promise<string> {
    // Check if we have a valid token
    if (this.accessToken && Date.now() < this.tokenExpiry) {
      return this.accessToken
    }

    try {
      const response = await fetch(`${this.config.baseUrl}/api/Auth/RequestToken`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Accept': 'application/json'
        },
        body: JSON.stringify({
          consumer_key: this.config.consumerKey,
          consumer_secret: this.config.consumerSecret
        })
      })

      const data = await response.json()
      
      if (data.token) {
        this.accessToken = data.token
        // Set expiry to 1 hour from now
        this.tokenExpiry = Date.now() + (60 * 60 * 1000)
        return this.accessToken
      } else {
        throw new Error('Failed to get access token')
      }
    } catch (error) {
      console.error('Pesapal token request failed:', error)
      throw new Error('Authentication failed')
    }
  }

  /**
   * Create a payment request with Pesapal
   */
  async createPayment(request: PesapalPaymentRequest): Promise<PesapalPaymentResponse> {
    try {
      const token = await this.getAccessToken()

      // In a real implementation, this would make an API call to Pesapal
      // For now, we'll simulate the integration
      
      const payload = {
        id: request.reference,
        currency: request.currency,
        amount: request.amount,
        description: request.description,
        callback_url: request.callbackUrl,
        notification_url: request.notificationUrl,
        cancellation_url: request.cancellationUrl,
        billing_address: {
          phone_number: request.phoneNumber,
          email_address: request.email,
          country_code: 'ZW',
          first_name: request.firstName,
          middle_name: '',
          last_name: request.lastName,
          line_1: '',
          line_2: '',
          city: '',
          state: '',
          postal_code: '',
          zip_code: ''
        }
      }

      // Simulate API call
      const response = await this.simulatePesapalApiCall(payload, token)
      
      return {
        success: true,
        orderTrackingId: response.order_tracking_id,
        merchantReference: response.merchant_reference,
        redirectUrl: response.redirect_url,
        message: 'Payment request created successfully'
      }
    } catch (error) {
      console.error('Pesapal payment creation failed:', error)
      return {
        success: false,
        error: error instanceof Error ? error.message : 'Payment creation failed'
      }
    }
  }

  /**
   * Get payment status from Pesapal
   */
  async getPaymentStatus(orderTrackingId: string): Promise<PesapalPaymentResponse> {
    try {
      const token = await this.getAccessToken()
      
      // In a real implementation, this would query Pesapal API
      const status = await this.simulateStatusCheck(orderTrackingId, token)
      
      return {
        success: true,
        message: `Payment status: ${status}`
      }
    } catch (error) {
      return {
        success: false,
        error: error instanceof Error ? error.message : 'Status check failed'
      }
    }
  }

  /**
   * Verify webhook signature from Pesapal
   */
  verifyWebhookSignature(payload: any, signature: string): boolean {
    // In a real implementation, you would verify the signature using HMAC
    // For now, we'll simulate verification
    return true
  }

  /**
   * Process webhook notification from Pesapal
   */
  async processWebhook(payload: PesapalWebhookPayload): Promise<{ success: boolean; error?: string }> {
    try {
      // Verify signature
      if (!this.verifyWebhookSignature(payload, payload.signature)) {
        return { success: false, error: 'Invalid signature' }
      }

      // Process the webhook based on status
      switch (payload.paymentStatus) {
        case 'COMPLETED':
          // Update transaction status in database
          await this.updateTransactionStatus(payload.orderTrackingId, 'completed')
          break
        case 'FAILED':
          await this.updateTransactionStatus(payload.orderTrackingId, 'failed')
          break
        case 'PENDING':
          // Keep as pending, no action needed
          break
      }

      return { success: true }
    } catch (error) {
      console.error('Pesapal webhook processing failed:', error)
      return { 
        success: false, 
        error: error instanceof Error ? error.message : 'Webhook processing failed' 
      }
    }
  }

  /**
   * Simulate Pesapal API call (replace with actual API integration)
   */
  private async simulatePesapalApiCall(payload: any, token: string): Promise<any> {
    // Simulate API delay
    await new Promise(resolve => setTimeout(resolve, 1000))
    
    return {
      order_tracking_id: `pesapal_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`,
      merchant_reference: payload.id,
      redirect_url: `https://pesapal.com/redirect/${payload.id}?amount=${payload.amount}&currency=${payload.currency}`,
      status: 'PENDING'
    }
  }

  /**
   * Simulate status check (replace with actual API call)
   */
  private async simulateStatusCheck(orderTrackingId: string, token: string): Promise<string> {
    await new Promise(resolve => setTimeout(resolve, 500))
    return 'COMPLETED'
  }

  /**
   * Update transaction status in database
   */
  private async updateTransactionStatus(orderTrackingId: string, status: string): Promise<void> {
    // In a real implementation, this would update the database
    console.log(`Updating Pesapal transaction ${orderTrackingId} status to ${status}`)
  }
}

// Default configuration (replace with actual credentials)
export const defaultPesapalConfig: PesapalConfig = {
  consumerKey: process.env.PESAPAL_CONSUMER_KEY || 'your_consumer_key',
  consumerSecret: process.env.PESAPAL_CONSUMER_SECRET || 'your_consumer_secret',
  baseUrl: process.env.PESAPAL_BASE_URL || 'https://api.pesapal.com',
  webhookSecret: process.env.PESAPAL_WEBHOOK_SECRET || 'your_webhook_secret',
  environment: (process.env.PESAPAL_ENVIRONMENT as 'sandbox' | 'production') || 'sandbox'
}

// Export service instance
export const pesapalService = new PesapalService(defaultPesapalConfig)
