// Performance optimization utilities and configurations

export const PERFORMANCE_CONFIG = {
  // Database query limits
  MAX_TRANSACTIONS_PER_PAGE: 20,
  MAX_BETS_PER_PAGE: 50,
  MAX_MARKETS_PER_PAGE: 30,
  
  // Debounce delays (ms)
  SEARCH_DEBOUNCE: 300,
  FORM_DEBOUNCE: 500,
  API_DEBOUNCE: 1000,
  
  // Throttle delays (ms)
  SCROLL_THROTTLE: 100,
  RESIZE_THROTTLE: 250,
  API_THROTTLE: 2000,
  
  // Real-time subscription intervals (ms)
  MARKET_UPDATE_INTERVAL: 5000,
  BALANCE_UPDATE_INTERVAL: 3000,
  ACTIVITY_UPDATE_INTERVAL: 2000,
  
  // Memory limits
  MAX_CACHED_ITEMS: 100,
  CACHE_EXPIRY_TIME: 5 * 60 * 1000, // 5 minutes
}

export const OPTIMIZATION_TIPS = {
  // Component optimization
  USE_MEMO: 'Use React.memo for components that receive stable props',
  USE_CALLBACK: 'Use useCallback for functions passed as props',
  USE_MEMO_HOOK: 'Use useMemo for expensive calculations',
  
  // Database optimization
  LIMIT_QUERIES: 'Always limit database queries with .limit()',
  SELECT_FIELDS: 'Only select necessary fields with .select()',
  USE_INDEXES: 'Ensure database indexes exist for filtered queries',
  
  // Real-time optimization
  DEBOUNCE_UPDATES: 'Debounce real-time updates to prevent excessive re-renders',
  FILTER_SUBSCRIPTIONS: 'Use filters on subscriptions to reduce unnecessary updates',
  CLEANUP_SUBSCRIPTIONS: 'Always unsubscribe from real-time subscriptions',
  
  // Bundle optimization
  CODE_SPLITTING: 'Use dynamic imports for large components',
  LAZY_LOADING: 'Implement lazy loading for images and components',
  TREE_SHAKING: 'Remove unused imports and dead code',
}

// Performance monitoring utilities
export function measurePerformance<T>(
  name: string,
  fn: () => T,
  threshold: number = 16 // 16ms for 60fps
): T {
  if (process.env.NODE_ENV === 'production') {
    return fn()
  }

  const start = performance.now()
  const result = fn()
  const end = performance.now()
  const duration = end - start

  if (duration > threshold) {
    console.warn(`Slow operation detected: ${name} took ${duration.toFixed(2)}ms`)
  }

  return result
}

// Memory usage monitoring
export function getMemoryUsage() {
  if (typeof performance !== 'undefined' && (performance as any).memory) {
    const memory = (performance as any).memory
    return {
      used: memory.usedJSHeapSize / 1024 / 1024, // MB
      total: memory.totalJSHeapSize / 1024 / 1024, // MB
      limit: memory.jsHeapSizeLimit / 1024 / 1024, // MB
    }
  }
  return null
}

// Cache management
class SimpleCache<T> {
  private cache = new Map<string, { data: T; timestamp: number }>()
  private maxSize: number
  private expiryTime: number

  constructor(maxSize = PERFORMANCE_CONFIG.MAX_CACHED_ITEMS, expiryTime = PERFORMANCE_CONFIG.CACHE_EXPIRY_TIME) {
    this.maxSize = maxSize
    this.expiryTime = expiryTime
  }

  set(key: string, data: T): void {
    // Remove expired entries
    this.cleanup()
    
    // Remove oldest entries if cache is full
    if (this.cache.size >= this.maxSize) {
      const oldestKey = this.cache.keys().next().value
      this.cache.delete(oldestKey)
    }

    this.cache.set(key, {
      data,
      timestamp: Date.now()
    })
  }

  get(key: string): T | null {
    const entry = this.cache.get(key)
    
    if (!entry) {
      return null
    }

    // Check if entry is expired
    if (Date.now() - entry.timestamp > this.expiryTime) {
      this.cache.delete(key)
      return null
    }

    return entry.data
  }

  has(key: string): boolean {
    return this.get(key) !== null
  }

  delete(key: string): void {
    this.cache.delete(key)
  }

  clear(): void {
    this.cache.clear()
  }

  private cleanup(): void {
    const now = Date.now()
    for (const [key, entry] of this.cache.entries()) {
      if (now - entry.timestamp > this.expiryTime) {
        this.cache.delete(key)
      }
    }
  }

  size(): number {
    return this.cache.size
  }
}

export const createCache = <T>(maxSize?: number, expiryTime?: number) => 
  new SimpleCache<T>(maxSize, expiryTime)

// Bundle size monitoring
export function getBundleSize() {
  if (typeof window !== 'undefined' && (window as any).__NEXT_DATA__) {
    const scripts = document.querySelectorAll('script[src]')
    let totalSize = 0
    
    scripts.forEach(script => {
      const src = script.getAttribute('src')
      if (src && src.includes('_next/static')) {
        // This is a simplified check - in reality you'd need to fetch the actual file sizes
        totalSize += 100 // Placeholder
      }
    })
    
    return totalSize
  }
  return 0
}
