import { supabase } from '@/lib/supabase-client'

export interface AccountDeletionData {
  userId: string
  confirmationText: string
  reason?: string
}

export class AccountDeletionService {
  private static instance: AccountDeletionService

  static getInstance(): AccountDeletionService {
    if (!AccountDeletionService.instance) {
      AccountDeletionService.instance = new AccountDeletionService()
    }
    return AccountDeletionService.instance
  }

  async deleteAccount(data: AccountDeletionData): Promise<{ success: boolean; error?: string }> {
    try {
      const { userId, confirmationText, reason } = data

      // Validate confirmation text
      if (confirmationText.toLowerCase() !== 'delete my account') {
        return {
          success: false,
          error: 'Confirmation text must be exactly "delete my account"'
        }
      }

      // Step 1: End all login sessions
      await this.endAllSessions(userId)

      // Step 2: Cancel all pending bets
      await this.cancelPendingBets(userId)

      // Step 3: Process any pending withdrawals
      await this.processPendingWithdrawals(userId)

      // Step 4: Create a record of the deletion request
      await this.createDeletionRecord(userId, reason)

      // Step 5: Delete user data from all tables
      await this.deleteUserData(userId)

      // Step 6: Delete the user from Supabase Auth
      const { error: authError } = await supabase.auth.admin.deleteUser(userId)

      if (authError) {
        console.error('Error deleting user from auth:', authError)
        // Continue with data deletion even if auth deletion fails
      }

      return { success: true }
    } catch (error) {
      console.error('Error in deleteAccount:', error)
      return {
        success: false,
        error: error instanceof Error ? error.message : 'Failed to delete account'
      }
    }
  }

  private async endAllSessions(userId: string): Promise<void> {
    try {
      const { error } = await supabase
        .from('login_sessions')
        .update({
          is_active: false,
          logout_time: new Date().toISOString()
        })
        .eq('user_id', userId)
        .eq('is_active', true)

      if (error) {
        console.error('Error ending sessions:', error)
      }
    } catch (error) {
      console.error('Error in endAllSessions:', error)
    }
  }

  private async cancelPendingBets(userId: string): Promise<void> {
    try {
      // Get all pending bets
      const { data: pendingBets, error: fetchError } = await supabase
        .from('bets')
        .select('*')
        .eq('user_id', userId)
        .in('status', ['seeded', 'live', 'matched'])

      if (fetchError) {
        console.error('Error fetching pending bets:', fetchError)
        return
      }

      // Cancel each pending bet and refund the amount
      for (const bet of pendingBets || []) {
        try {
          // Update bet status to cancelled
          await supabase
            .from('bets')
            .update({ status: 'cancelled' })
            .eq('id', bet.id)

          // Add refund transaction
          await supabase
            .from('unified_transactions')
            .insert({
              user_id: userId,
              type: 'refund',
              amount: bet.amount,
              description: `Refund for cancelled bet on market ${bet.market_id}`,
              status: 'completed',
              market_id: bet.market_id
            })

          // Update user balance
          await supabase
            .from('profiles')
            .update({
              balance: supabase.sql`balance + ${bet.amount}`
            })
            .eq('id', userId)
        } catch (error) {
          console.error(`Error cancelling bet ${bet.id}:`, error)
        }
      }
    } catch (error) {
      console.error('Error in cancelPendingBets:', error)
    }
  }

  private async processPendingWithdrawals(userId: string): Promise<void> {
    try {
      // Get all pending withdrawals
      const { data: pendingWithdrawals, error: fetchError } = await supabase
        .from('unified_transactions')
        .select('*')
        .eq('user_id', userId)
        .eq('type', 'withdrawal')
        .eq('status', 'pending')

      if (fetchError) {
        console.error('Error fetching pending withdrawals:', fetchError)
        return
      }

      // Cancel pending withdrawals and refund to balance
      for (const withdrawal of pendingWithdrawals || []) {
        try {
          // Update withdrawal status to cancelled
          await supabase
            .from('unified_transactions')
            .update({ status: 'cancelled' })
            .eq('id', withdrawal.id)

          // Add refund transaction
          await supabase
            .from('unified_transactions')
            .insert({
              user_id: userId,
              type: 'refund',
              amount: withdrawal.amount,
              description: 'Refund for cancelled withdrawal',
              status: 'completed'
            })

          // Update user balance
          await supabase
            .from('profiles')
            .update({
              balance: supabase.sql`balance + ${withdrawal.amount}`
            })
            .eq('id', userId)
        } catch (error) {
          console.error(`Error processing withdrawal ${withdrawal.id}:`, error)
        }
      }
    } catch (error) {
      console.error('Error in processPendingWithdrawals:', error)
    }
  }

  private async createDeletionRecord(userId: string, reason?: string): Promise<void> {
    try {
      await supabase
        .from('account_deletions')
        .insert({
          user_id: userId,
          reason: reason || 'User requested account deletion',
          deleted_at: new Date().toISOString()
        })
    } catch (error) {
      console.error('Error creating deletion record:', error)
      // Continue even if this fails
    }
  }

  private async deleteUserData(userId: string): Promise<void> {
    try {
      // Delete in order of dependencies
      const tables = [
        'login_sessions',
        'notifications',
        'unified_transactions',
        'bets',
        'user_preferences',
        'payment_methods',
        'profiles'
      ]

      for (const table of tables) {
        try {
          const { error } = await supabase
            .from(table)
            .delete()
            .eq('user_id', userId)

          if (error && !error.message.includes('does not exist')) {
            console.error(`Error deleting from ${table}:`, error)
          }
        } catch (error) {
          console.error(`Error deleting from ${table}:`, error)
        }
      }
    } catch (error) {
      console.error('Error in deleteUserData:', error)
    }
  }

  async validateAccountDeletion(userId: string): Promise<{
    canDelete: boolean
    warnings: string[]
    pendingTransactions: number
    activeBets: number
  }> {
    try {
      const warnings: string[] = []
      let pendingTransactions = 0
      let activeBets = 0

      // Check for pending withdrawals
      const { count: withdrawalCount } = await supabase
        .from('unified_transactions')
        .select('*', { count: 'exact', head: true })
        .eq('user_id', userId)
        .eq('type', 'withdrawal')
        .eq('status', 'pending')

      if (withdrawalCount && withdrawalCount > 0) {
        pendingTransactions += withdrawalCount
        warnings.push(`${withdrawalCount} pending withdrawal(s) will be cancelled and refunded`)
      }

      // Check for active bets
      const { count: betCount } = await supabase
        .from('bets')
        .select('*', { count: 'exact', head: true })
        .eq('user_id', userId)
        .in('status', ['seeded', 'live', 'matched'])

      if (betCount && betCount > 0) {
        activeBets += betCount
        warnings.push(`${betCount} active bet(s) will be cancelled and refunded`)
      }

      // Check account balance
      const { data: profile } = await supabase
        .from('profiles')
        .select('balance')
        .eq('id', userId)
        .single()

      if (profile && profile.balance > 0) {
        warnings.push(`Account balance of $${profile.balance.toFixed(2)} will be forfeited`)
      }

      return {
        canDelete: true,
        warnings,
        pendingTransactions,
        activeBets
      }
    } catch (error) {
      console.error('Error validating account deletion:', error)
      return {
        canDelete: false,
        warnings: ['Unable to validate account deletion'],
        pendingTransactions: 0,
        activeBets: 0
      }
    }
  }
}

// Export singleton instance
export const accountDeletionService = AccountDeletionService.getInstance()
