// Dynamic import for server-side only
let nodemailer: any = null

export interface EmailConfig {
  host: string
  port: number
  secure: boolean
  auth: {
    user: string
    pass: string
  }
}

export interface EmailTemplate {
  subject: string
  html: string
  text: string
}

export interface EmailData {
  to: string
  subject: string
  html: string
  text: string
  from?: string
}

export class EmailService {
  private transporter: any = null
  private config: EmailConfig

  constructor(config: EmailConfig) {
    this.config = config
    // Initialize transporter only on server side
    if (typeof window === 'undefined') {
      this.initializeTransporter()
    }
  }

  private async initializeTransporter() {
    if (typeof window !== 'undefined') return // Skip on client side
    
    try {
      nodemailer = await import('nodemailer')
      this.transporter = nodemailer.createTransporter({
        host: this.config.host,
        port: this.config.port,
        secure: this.config.secure,
        auth: {
          user: this.config.auth.user,
          pass: this.config.auth.pass
        }
      })
    } catch (error) {
      console.error('Failed to initialize email transporter:', error)
    }
  }

  /**
   * Send an email
   */
  async sendEmail(emailData: EmailData): Promise<boolean> {
    try {
      // Ensure transporter is initialized
      if (!this.transporter) {
        await this.initializeTransporter()
      }
      
      if (!this.transporter) {
        console.error('Email transporter not available')
        return false
      }

      const mailOptions = {
        from: emailData.from || this.config.auth.user,
        to: emailData.to,
        subject: emailData.subject,
        html: emailData.html,
        text: emailData.text
      }

      const result = await this.transporter.sendMail(mailOptions)
      console.log('Email sent successfully:', result.messageId)
      return true
    } catch (error) {
      console.error('Error sending email:', error)
      return false
    }
  }

  /**
   * Send welcome email to new user
   */
  async sendWelcomeEmail(userEmail: string, userName: string): Promise<boolean> {
    const template = this.getWelcomeEmailTemplate(userName)
    
    return this.sendEmail({
      to: userEmail,
      subject: template.subject,
      html: template.html,
      text: template.text
    })
  }

  /**
   * Send bet result email
   */
  async sendBetResultEmail(
    userEmail: string, 
    userName: string, 
    outcome: 'won' | 'lost',
    amount: number,
    marketTitle: string
  ): Promise<boolean> {
    const template = this.getBetResultEmailTemplate(userName, outcome, amount, marketTitle)
    
    return this.sendEmail({
      to: userEmail,
      subject: template.subject,
      html: template.html,
      text: template.text
    })
  }

  /**
   * Send deposit confirmation email
   */
  async sendDepositConfirmationEmail(
    userEmail: string,
    userName: string,
    amount: number,
    transactionId: string
  ): Promise<boolean> {
    const template = this.getDepositConfirmationTemplate(userName, amount, transactionId)
    
    return this.sendEmail({
      to: userEmail,
      subject: template.subject,
      html: template.html,
      text: template.text
    })
  }

  /**
   * Send withdrawal confirmation email
   */
  async sendWithdrawalConfirmationEmail(
    userEmail: string,
    userName: string,
    amount: number,
    transactionId: string
  ): Promise<boolean> {
    const template = this.getWithdrawalConfirmationTemplate(userName, amount, transactionId)
    
    return this.sendEmail({
      to: userEmail,
      subject: template.subject,
      html: template.html,
      text: template.text
    })
  }

  /**
   * Send market resolution email
   */
  async sendMarketResolutionEmail(
    userEmail: string,
    userName: string,
    marketTitle: string,
    winningOutcome: string,
    userOutcome: string,
    won: boolean,
    amount: number
  ): Promise<boolean> {
    const template = this.getMarketResolutionTemplate(
      userName, 
      marketTitle, 
      winningOutcome, 
      userOutcome, 
      won, 
      amount
    )
    
    return this.sendEmail({
      to: userEmail,
      subject: template.subject,
      html: template.html,
      text: template.text
    })
  }

  /**
   * Send KYC status email
   */
  async sendKYCStatusEmail(
    userEmail: string, 
    userName: string, 
    status: 'approved' | 'rejected', 
    reason?: string
  ): Promise<boolean> {
    const template = this.getKYCStatusTemplate(userName, status, reason)
    
    return this.sendEmail({
      to: userEmail,
      subject: template.subject,
      html: template.html,
      text: template.text
    })
  }

  async sendRefundEmail(
    userEmail: string,
    userName: string,
    amount: number,
    betId: string,
    outcome: string,
    marketTitle: string
  ): Promise<boolean> {
    const template = this.getRefundEmailTemplate(userName, amount, betId, outcome, marketTitle)
    
    return this.sendEmail({
      to: userEmail,
      subject: template.subject,
      html: template.html,
      text: template.text
    })
  }

  async sendMarketRefundEmail(
    userEmail: string,
    userName: string,
    amount: number,
    marketId: string,
    betCount: number
  ): Promise<boolean> {
    const template = this.getMarketRefundEmailTemplate(userName, amount, marketId, betCount)
    
    return this.sendEmail({
      to: userEmail,
      subject: template.subject,
      html: template.html,
      text: template.text
    })
  }

  /**
   * Get welcome email template
   */
  private getWelcomeEmailTemplate(userName: string): EmailTemplate {
    const subject = 'Welcome to BetFusion Zim! 🎉'
    
    const html = `
      <!DOCTYPE html>
      <html>
      <head>
        <meta charset="utf-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <title>Welcome to BetFusion Zim</title>
        <style>
          body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
          .container { max-width: 600px; margin: 0 auto; padding: 20px; }
          .header { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; padding: 30px; text-align: center; border-radius: 10px 10px 0 0; }
          .content { background: #f9f9f9; padding: 30px; border-radius: 0 0 10px 10px; }
          .button { display: inline-block; background: #667eea; color: white; padding: 12px 30px; text-decoration: none; border-radius: 5px; margin: 20px 0; }
          .footer { text-align: center; margin-top: 30px; color: #666; font-size: 14px; }
        </style>
      </head>
      <body>
        <div class="container">
          <div class="header">
            <h1>🎉 Welcome to BetFusion Zim!</h1>
            <p>Your prediction market journey starts here</p>
          </div>
          <div class="content">
            <h2>Hello ${userName}!</h2>
            <p>Welcome to BetFusion Zim, Zimbabwe's premier prediction market platform! We're excited to have you join our community of smart bettors and market predictors.</p>
            
            <h3>🚀 What you can do:</h3>
            <ul>
              <li><strong>Bet on Markets:</strong> Predict outcomes on various events and earn rewards</li>
              <li><strong>Create Markets:</strong> Start your own prediction markets</li>
              <li><strong>Track Performance:</strong> Monitor your betting history and performance</li>
              <li><strong>Join Community:</strong> Connect with other predictors and share insights</li>
            </ul>
            
            <p>Ready to start? Your account is all set up and ready to go!</p>
            
            <a href="${process.env.NEXT_PUBLIC_SITE_URL || 'http://localhost:3000'}" class="button">Start Betting Now</a>
            
            <h3>💡 Quick Tips:</h3>
            <ul>
              <li>Start with small bets to get familiar with the platform</li>
              <li>Read market descriptions carefully before placing bets</li>
              <li>Check your email regularly for important updates</li>
              <li>Join our community discussions for market insights</li>
            </ul>
          </div>
          <div class="footer">
            <p>Best regards,<br>The BetFusion Zim Team</p>
            <p>This email was sent to ${userEmail}. If you didn't create an account, please ignore this email.</p>
          </div>
        </div>
      </body>
      </html>
    `

    const text = `
Welcome to BetFusion Zim!

Hello ${userName}!

Welcome to BetFusion Zim, Zimbabwe's premier prediction market platform! We're excited to have you join our community of smart bettors and market predictors.

What you can do:
- Bet on Markets: Predict outcomes on various events and earn rewards
- Create Markets: Start your own prediction markets  
- Track Performance: Monitor your betting history and performance
- Join Community: Connect with other predictors and share insights

Ready to start? Your account is all set up and ready to go!

Visit: ${process.env.NEXT_PUBLIC_SITE_URL || 'http://localhost:3000'}

Quick Tips:
- Start with small bets to get familiar with the platform
- Read market descriptions carefully before placing bets
- Check your email regularly for important updates
- Join our community discussions for market insights

Best regards,
The BetFusion Zim Team

This email was sent to ${userEmail}. If you didn't create an account, please ignore this email.
    `

    return { subject, html, text }
  }

  /**
   * Get bet result email template
   */
  private getBetResultEmailTemplate(
    userName: string, 
    outcome: 'won' | 'lost',
    amount: number,
    marketTitle: string
  ): EmailTemplate {
    const isWin = outcome === 'won'
    const subject = isWin ? `🎉 You Won $${amount.toFixed(2)} on ${marketTitle}!` : `📉 Bet Result: ${marketTitle}`
    
    const html = `
      <!DOCTYPE html>
      <html>
      <head>
        <meta charset="utf-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <title>Bet Result - ${marketTitle}</title>
        <style>
          body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
          .container { max-width: 600px; margin: 0 auto; padding: 20px; }
          .header { background: ${isWin ? 'linear-gradient(135deg, #10b981 0%, #059669 100%)' : 'linear-gradient(135deg, #ef4444 0%, #dc2626 100%)'}; color: white; padding: 30px; text-align: center; border-radius: 10px 10px 0 0; }
          .content { background: #f9f9f9; padding: 30px; border-radius: 0 0 10px 10px; }
          .result-box { background: ${isWin ? '#d1fae5' : '#fee2e2'}; border: 2px solid ${isWin ? '#10b981' : '#ef4444'}; padding: 20px; border-radius: 8px; margin: 20px 0; text-align: center; }
          .button { display: inline-block; background: #667eea; color: white; padding: 12px 30px; text-decoration: none; border-radius: 5px; margin: 20px 0; }
          .footer { text-align: center; margin-top: 30px; color: #666; font-size: 14px; }
        </style>
      </head>
      <body>
        <div class="container">
          <div class="header">
            <h1>${isWin ? '🎉 Congratulations!' : '📉 Bet Result'}</h1>
            <p>${isWin ? 'You won your bet!' : 'Your bet result is in'}</p>
          </div>
          <div class="content">
            <h2>Hello ${userName}!</h2>
            
            <div class="result-box">
              <h3>${isWin ? '🎉 You Won!' : '📉 You Lost'}</h3>
              <p><strong>Market:</strong> ${marketTitle}</p>
              <p><strong>Amount:</strong> $${amount.toFixed(2)}</p>
              <p><strong>Result:</strong> ${isWin ? 'Won' : 'Lost'}</p>
            </div>
            
            ${isWin ? `
              <p>Congratulations! Your prediction was correct and you've won $${amount.toFixed(2)}! The funds have been added to your account balance.</p>
              <p>Keep up the great work and continue making smart predictions!</p>
            ` : `
              <p>Unfortunately, your prediction didn't work out this time. Don't worry - every great predictor has ups and downs!</p>
              <p>Keep analyzing the markets and trust your instincts for the next bet.</p>
            `}
            
            <a href="${process.env.NEXT_PUBLIC_SITE_URL || 'http://localhost:3000'}/portfolio" class="button">View Your Portfolio</a>
            
            <h3>💡 What's Next?</h3>
            <ul>
              <li>Check out new markets and opportunities</li>
              <li>Review your betting strategy</li>
              <li>Join community discussions for insights</li>
              <li>Create your own prediction markets</li>
            </ul>
          </div>
          <div class="footer">
            <p>Best regards,<br>The BetFusion Zim Team</p>
          </div>
        </div>
      </body>
      </html>
    `

    const text = `
${isWin ? '🎉 Congratulations! You Won!' : '📉 Bet Result'}

Hello ${userName}!

${isWin ? 'Congratulations! Your prediction was correct and you\'ve won $' + amount.toFixed(2) + '! The funds have been added to your account balance.' : 'Unfortunately, your prediction didn\'t work out this time. Don\'t worry - every great predictor has ups and downs!'}

Market: ${marketTitle}
Amount: $${amount.toFixed(2)}
Result: ${isWin ? 'Won' : 'Lost'}

${isWin ? 'Keep up the great work and continue making smart predictions!' : 'Keep analyzing the markets and trust your instincts for the next bet.'}

What's Next?
- Check out new markets and opportunities
- Review your betting strategy  
- Join community discussions for insights
- Create your own prediction markets

Visit your portfolio: ${process.env.NEXT_PUBLIC_SITE_URL || 'http://localhost:3000'}/portfolio

Best regards,
The BetFusion Zim Team
    `

    return { subject, html, text }
  }

  /**
   * Get deposit confirmation template
   */
  private getDepositConfirmationTemplate(
    userName: string,
    amount: number,
    transactionId: string
  ): EmailTemplate {
    const subject = `💰 Deposit Confirmation - $${amount.toFixed(2)}`
    
    const html = `
      <!DOCTYPE html>
      <html>
      <head>
        <meta charset="utf-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <title>Deposit Confirmation</title>
        <style>
          body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
          .container { max-width: 600px; margin: 0 auto; padding: 20px; }
          .header { background: linear-gradient(135deg, #10b981 0%, #059669 100%); color: white; padding: 30px; text-align: center; border-radius: 10px 10px 0 0; }
          .content { background: #f9f9f9; padding: 30px; border-radius: 0 0 10px 10px; }
          .transaction-box { background: #d1fae5; border: 2px solid #10b981; padding: 20px; border-radius: 8px; margin: 20px 0; }
          .button { display: inline-block; background: #667eea; color: white; padding: 12px 30px; text-decoration: none; border-radius: 5px; margin: 20px 0; }
          .footer { text-align: center; margin-top: 30px; color: #666; font-size: 14px; }
        </style>
      </head>
      <body>
        <div class="container">
          <div class="header">
            <h1>💰 Deposit Confirmed!</h1>
            <p>Your funds have been added to your account</p>
          </div>
          <div class="content">
            <h2>Hello ${userName}!</h2>
            
            <div class="transaction-box">
              <h3>💰 Deposit Details</h3>
              <p><strong>Amount:</strong> $${amount.toFixed(2)}</p>
              <p><strong>Transaction ID:</strong> ${transactionId}</p>
              <p><strong>Status:</strong> Completed</p>
              <p><strong>Date:</strong> ${new Date().toLocaleDateString()}</p>
            </div>
            
            <p>Your deposit has been successfully processed and the funds are now available in your account. You can start betting immediately!</p>
            
            <a href="${process.env.NEXT_PUBLIC_SITE_URL || 'http://localhost:3000'}/wallet" class="button">View Your Wallet</a>
            
            <h3>🚀 Ready to Bet?</h3>
            <ul>
              <li>Browse available markets</li>
              <li>Place your first bet</li>
              <li>Create your own markets</li>
              <li>Join community discussions</li>
            </ul>
          </div>
          <div class="footer">
            <p>Best regards,<br>The BetFusion Zim Team</p>
          </div>
        </div>
      </body>
      </html>
    `

    const text = `
💰 Deposit Confirmation

Hello ${userName}!

Your deposit has been successfully processed and the funds are now available in your account.

Deposit Details:
- Amount: $${amount.toFixed(2)}
- Transaction ID: ${transactionId}
- Status: Completed
- Date: ${new Date().toLocaleDateString()}

You can start betting immediately!

Ready to Bet?
- Browse available markets
- Place your first bet
- Create your own markets
- Join community discussions

View your wallet: ${process.env.NEXT_PUBLIC_SITE_URL || 'http://localhost:3000'}/wallet

Best regards,
The BetFusion Zim Team
    `

    return { subject, html, text }
  }

  /**
   * Get withdrawal confirmation template
   */
  private getWithdrawalConfirmationTemplate(
    userName: string,
    amount: number,
    transactionId: string
  ): EmailTemplate {
    const subject = `💸 Withdrawal Confirmation - $${amount.toFixed(2)}`
    
    const html = `
      <!DOCTYPE html>
      <html>
      <head>
        <meta charset="utf-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <title>Withdrawal Confirmation</title>
        <style>
          body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
          .container { max-width: 600px; margin: 0 auto; padding: 20px; }
          .header { background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%); color: white; padding: 30px; text-align: center; border-radius: 10px 10px 0 0; }
          .content { background: #f9f9f9; padding: 30px; border-radius: 0 0 10px 10px; }
          .transaction-box { background: #fef3c7; border: 2px solid #f59e0b; padding: 20px; border-radius: 8px; margin: 20px 0; }
          .button { display: inline-block; background: #667eea; color: white; padding: 12px 30px; text-decoration: none; border-radius: 5px; margin: 20px 0; }
          .footer { text-align: center; margin-top: 30px; color: #666; font-size: 14px; }
        </style>
      </head>
      <body>
        <div class="container">
          <div class="header">
            <h1>💸 Withdrawal Confirmed!</h1>
            <p>Your withdrawal request has been processed</p>
          </div>
          <div class="content">
            <h2>Hello ${userName}!</h2>
            
            <div class="transaction-box">
              <h3>💸 Withdrawal Details</h3>
              <p><strong>Amount:</strong> $${amount.toFixed(2)}</p>
              <p><strong>Transaction ID:</strong> ${transactionId}</p>
              <p><strong>Status:</strong> Processing</p>
              <p><strong>Date:</strong> ${new Date().toLocaleDateString()}</p>
            </div>
            
            <p>Your withdrawal request has been submitted and is being processed. Funds will be transferred to your account within 1-3 business days.</p>
            
            <a href="${process.env.NEXT_PUBLIC_SITE_URL || 'http://localhost:3000'}/wallet" class="button">View Your Wallet</a>
            
            <h3>📋 Important Notes:</h3>
            <ul>
              <li>Processing time: 1-3 business days</li>
              <li>You'll receive another email when the transfer is complete</li>
              <li>Contact support if you have any questions</li>
            </ul>
          </div>
          <div class="footer">
            <p>Best regards,<br>The BetFusion Zim Team</p>
          </div>
        </div>
      </body>
      </html>
    `

    const text = `
💸 Withdrawal Confirmation

Hello ${userName}!

Your withdrawal request has been submitted and is being processed.

Withdrawal Details:
- Amount: $${amount.toFixed(2)}
- Transaction ID: ${transactionId}
- Status: Processing
- Date: ${new Date().toLocaleDateString()}

Funds will be transferred to your account within 1-3 business days.

Important Notes:
- Processing time: 1-3 business days
- You'll receive another email when the transfer is complete
- Contact support if you have any questions

View your wallet: ${process.env.NEXT_PUBLIC_SITE_URL || 'http://localhost:3000'}/wallet

Best regards,
The BetFusion Zim Team
    `

    return { subject, html, text }
  }

  /**
   * Get market resolution template
   */
  private getMarketResolutionTemplate(
    userName: string,
    marketTitle: string,
    winningOutcome: string,
    userOutcome: string,
    won: boolean,
    amount: number
  ): EmailTemplate {
    const subject = `🏁 Market Resolved: ${marketTitle}`
    
    const html = `
      <!DOCTYPE html>
      <html>
      <head>
        <meta charset="utf-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <title>Market Resolution - ${marketTitle}</title>
        <style>
          body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
          .container { max-width: 600px; margin: 0 auto; padding: 20px; }
          .header { background: linear-gradient(135deg, #8b5cf6 0%, #7c3aed 100%); color: white; padding: 30px; text-align: center; border-radius: 10px 10px 0 0; }
          .content { background: #f9f9f9; padding: 30px; border-radius: 0 0 10px 10px; }
          .result-box { background: ${won ? '#d1fae5' : '#fee2e2'}; border: 2px solid ${won ? '#10b981' : '#ef4444'}; padding: 20px; border-radius: 8px; margin: 20px 0; }
          .button { display: inline-block; background: #667eea; color: white; padding: 12px 30px; text-decoration: none; border-radius: 5px; margin: 20px 0; }
          .footer { text-align: center; margin-top: 30px; color: #666; font-size: 14px; }
        </style>
      </head>
      <body>
        <div class="container">
          <div class="header">
            <h1>🏁 Market Resolved!</h1>
            <p>The market has been closed and results are in</p>
          </div>
          <div class="content">
            <h2>Hello ${userName}!</h2>
            
            <div class="result-box">
              <h3>🏁 Market Results</h3>
              <p><strong>Market:</strong> ${marketTitle}</p>
              <p><strong>Winning Outcome:</strong> ${winningOutcome}</p>
              <p><strong>Your Prediction:</strong> ${userOutcome}</p>
              <p><strong>Result:</strong> ${won ? '🎉 You Won!' : '📉 You Lost'}</p>
              ${won ? `<p><strong>Winnings:</strong> $${amount.toFixed(2)}</p>` : ''}
            </div>
            
            <p>The market "${marketTitle}" has been resolved. ${won ? 'Congratulations on your winning prediction!' : 'Better luck next time!'}</p>
            
            <a href="${process.env.NEXT_PUBLIC_SITE_URL || 'http://localhost:3000'}/portfolio" class="button">View Your Portfolio</a>
          </div>
          <div class="footer">
            <p>Best regards,<br>The BetFusion Zim Team</p>
          </div>
        </div>
      </body>
      </html>
    `

    const text = `
🏁 Market Resolved!

Hello ${userName}!

The market "${marketTitle}" has been resolved.

Market Results:
- Market: ${marketTitle}
- Winning Outcome: ${winningOutcome}
- Your Prediction: ${userOutcome}
- Result: ${won ? '🎉 You Won!' : '📉 You Lost'}
${won ? `- Winnings: $${amount.toFixed(2)}` : ''}

${won ? 'Congratulations on your winning prediction!' : 'Better luck next time!'}

View your portfolio: ${process.env.NEXT_PUBLIC_SITE_URL || 'http://localhost:3000'}/portfolio

Best regards,
The BetFusion Zim Team
    `

    return { subject, html, text }
  }

  /**
   * Get KYC status template
   */
  private getKYCStatusTemplate(
    userName: string,
    status: 'approved' | 'rejected',
    reason?: string
  ): EmailTemplate {
    const isApproved = status === 'approved'
    const subject = isApproved ? '✅ KYC Verification Approved!' : '❌ KYC Verification Update'
    
    const html = `
      <!DOCTYPE html>
      <html>
      <head>
        <meta charset="utf-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <title>KYC ${isApproved ? 'Approved' : 'Update'}</title>
        <style>
          body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
          .container { max-width: 600px; margin: 0 auto; padding: 20px; }
          .header { background: ${isApproved ? 'linear-gradient(135deg, #10b981 0%, #059669 100%)' : 'linear-gradient(135deg, #ef4444 0%, #dc2626 100%)'}; color: white; padding: 30px; text-align: center; border-radius: 10px 10px 0 0; }
          .content { background: #f9f9f9; padding: 30px; border-radius: 0 0 10px 10px; }
          .status-box { background: ${isApproved ? '#d1fae5' : '#fee2e2'}; border: 2px solid ${isApproved ? '#10b981' : '#ef4444'}; padding: 20px; border-radius: 8px; margin: 20px 0; }
          .button { display: inline-block; background: #667eea; color: white; padding: 12px 30px; text-decoration: none; border-radius: 5px; margin: 20px 0; }
          .footer { text-align: center; margin-top: 30px; color: #666; font-size: 14px; }
        </style>
      </head>
      <body>
        <div class="container">
          <div class="header">
            <h1>${isApproved ? '✅ KYC Approved!' : '❌ KYC Update'}</h1>
            <p>${isApproved ? 'Your identity has been verified' : 'Your KYC status has been updated'}</p>
          </div>
          <div class="content">
            <h2>Hello ${userName}!</h2>
            
            <div class="status-box">
              <h3>${isApproved ? '✅ Verification Approved' : '❌ Verification Update'}</h3>
              <p><strong>Status:</strong> ${isApproved ? 'Approved' : 'Rejected'}</p>
              ${!isApproved && reason ? `<p><strong>Reason:</strong> ${reason}</p>` : ''}
              <p><strong>Date:</strong> ${new Date().toLocaleDateString()}</p>
            </div>
            
            ${isApproved ? `
              <p>Congratulations! Your identity has been successfully verified. You now have full access to all platform features including higher withdrawal limits and enhanced security.</p>
              <p>You can now enjoy all the benefits of a verified account!</p>
            ` : `
              <p>We were unable to verify your identity at this time. ${reason ? `Reason: ${reason}` : 'Please ensure all documents are clear and valid.'}</p>
              <p>You can resubmit your KYC documents for review.</p>
            `}
            
            <a href="${process.env.NEXT_PUBLIC_SITE_URL || 'http://localhost:3000'}/kyc" class="button">${isApproved ? 'View Your Profile' : 'Resubmit KYC'}</a>
          </div>
          <div class="footer">
            <p>Best regards,<br>The BetFusion Zim Team</p>
          </div>
        </div>
      </body>
      </html>
    `

    const text = `
${isApproved ? '✅ KYC Approved!' : '❌ KYC Update'}

Hello ${userName}!

${isApproved ? 'Congratulations! Your identity has been successfully verified. You now have full access to all platform features including higher withdrawal limits and enhanced security.' : 'We were unable to verify your identity at this time. ' + (reason ? `Reason: ${reason}` : 'Please ensure all documents are clear and valid.')}

Status: ${isApproved ? 'Approved' : 'Rejected'}
${!isApproved && reason ? `Reason: ${reason}` : ''}
Date: ${new Date().toLocaleDateString()}

${isApproved ? 'You can now enjoy all the benefits of a verified account!' : 'You can resubmit your KYC documents for review.'}

${isApproved ? 'View your profile' : 'Resubmit KYC'}: ${process.env.NEXT_PUBLIC_SITE_URL || 'http://localhost:3000'}/kyc

Best regards,
The BetFusion Zim Team
    `

    return { subject, html, text }
  }

  /**
   * Get refund email template
   */
  private getRefundEmailTemplate(
    userName: string,
    amount: number,
    betId: string,
    outcome: string,
    marketTitle: string
  ): EmailTemplate {
    const subject = `💰 Bet Refund Processed - $${amount.toFixed(2)}`
    
    const html = `
      <!DOCTYPE html>
      <html>
      <head>
        <meta charset="utf-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <title>Bet Refund Processed</title>
        <style>
          body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
          .container { max-width: 600px; margin: 0 auto; padding: 20px; }
          .header { background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%); color: white; padding: 30px; text-align: center; border-radius: 10px 10px 0 0; }
          .content { background: #f9f9f9; padding: 30px; border-radius: 0 0 10px 10px; }
          .refund-box { background: #fef3c7; border: 2px solid #f59e0b; padding: 20px; border-radius: 8px; margin: 20px 0; }
          .button { display: inline-block; background: #667eea; color: white; padding: 12px 30px; text-decoration: none; border-radius: 5px; margin: 20px 0; }
          .footer { text-align: center; margin-top: 30px; color: #666; font-size: 14px; }
        </style>
      </head>
      <body>
        <div class="container">
          <div class="header">
            <h1>💰 Bet Refund Processed</h1>
            <p>Your bet has been refunded successfully</p>
          </div>
          <div class="content">
            <h2>Hello ${userName}!</h2>
            <p>Your bet has been refunded and the amount has been added back to your account balance.</p>
            
            <div class="refund-box">
              <h3>Refund Details</h3>
              <p><strong>Refund Amount:</strong> $${amount.toFixed(2)}</p>
              <p><strong>Bet ID:</strong> ${betId}</p>
              <p><strong>Outcome:</strong> ${outcome}</p>
              <p><strong>Market:</strong> ${marketTitle}</p>
            </div>
            
            <div style="background: #d1fae5; border: 2px solid #10b981; padding: 15px; border-radius: 8px; margin: 20px 0;">
              <p style="margin: 0; color: #166534;"><strong>✅ Refund Complete</strong><br>
              The refunded amount is now available in your account balance and can be used for new bets or withdrawals.</p>
            </div>
            
            <div style="text-align: center;">
              <a href="${process.env.NEXT_PUBLIC_SITE_URL || 'https://betfusionzim.com'}/wallet" class="button">View Your Wallet</a>
            </div>
            
            <p>If you have any questions about this refund, please contact our support team.</p>
          </div>
          <div class="footer">
            <p>© 2024 BetFusion Zim. All rights reserved.</p>
            <p>This is an automated message. Please do not reply to this email.</p>
          </div>
        </div>
      </body>
      </html>
    `
    
    const text = `
      Bet Refund Processed - BetFusion Zim
      
      Hello ${userName}!
      
      Your bet has been refunded and the amount has been added back to your account balance.
      
      Refund Details:
      - Refund Amount: $${amount.toFixed(2)}
      - Bet ID: ${betId}
      - Outcome: ${outcome}
      - Market: ${marketTitle}
      
      The refunded amount is now available in your account balance and can be used for new bets or withdrawals.
      
      View Your Wallet: ${process.env.NEXT_PUBLIC_SITE_URL || 'https://betfusionzim.com'}/wallet
      
      If you have any questions about this refund, please contact our support team.
      
      © 2024 BetFusion Zim. All rights reserved.
      This is an automated message. Please do not reply to this email.
    `
    
    return { subject, html, text }
  }

  /**
   * Get market refund email template
   */
  private getMarketRefundEmailTemplate(
    userName: string,
    amount: number,
    marketId: string,
    betCount: number
  ): EmailTemplate {
    const subject = `💰 Market Refund Processed - $${amount.toFixed(2)}`
    
    const html = `
      <!DOCTYPE html>
      <html>
      <head>
        <meta charset="utf-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <title>Market Refund Processed</title>
        <style>
          body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
          .container { max-width: 600px; margin: 0 auto; padding: 20px; }
          .header { background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%); color: white; padding: 30px; text-align: center; border-radius: 10px 10px 0 0; }
          .content { background: #f9f9f9; padding: 30px; border-radius: 0 0 10px 10px; }
          .refund-box { background: #fef3c7; border: 2px solid #f59e0b; padding: 20px; border-radius: 8px; margin: 20px 0; }
          .button { display: inline-block; background: #667eea; color: white; padding: 12px 30px; text-decoration: none; border-radius: 5px; margin: 20px 0; }
          .footer { text-align: center; margin-top: 30px; color: #666; font-size: 14px; }
        </style>
      </head>
      <body>
        <div class="container">
          <div class="header">
            <h1>💰 Market Refund Processed</h1>
            <p>Your market bets have been refunded successfully</p>
          </div>
          <div class="content">
            <h2>Hello ${userName}!</h2>
            <p>All your bets in this market have been refunded and the total amount has been added back to your account balance.</p>
            
            <div class="refund-box">
              <h3>Market Refund Details</h3>
              <p><strong>Total Refund Amount:</strong> $${amount.toFixed(2)}</p>
              <p><strong>Market ID:</strong> ${marketId}</p>
              <p><strong>Number of Bets Refunded:</strong> ${betCount}</p>
            </div>
            
            <div style="background: #d1fae5; border: 2px solid #10b981; padding: 15px; border-radius: 8px; margin: 20px 0;">
              <p style="margin: 0; color: #166534;"><strong>✅ Market Refund Complete</strong><br>
              All your bets in this market have been refunded. The total amount is now available in your account balance and can be used for new bets or withdrawals.</p>
            </div>
            
            <div style="text-align: center;">
              <a href="${process.env.NEXT_PUBLIC_SITE_URL || 'https://betfusionzim.com'}/wallet" class="button">View Your Wallet</a>
            </div>
            
            <p>If you have any questions about this market refund, please contact our support team.</p>
          </div>
          <div class="footer">
            <p>© 2024 BetFusion Zim. All rights reserved.</p>
            <p>This is an automated message. Please do not reply to this email.</p>
          </div>
        </div>
      </body>
      </html>
    `
    
    const text = `
      Market Refund Processed - BetFusion Zim
      
      Hello ${userName}!
      
      All your bets in this market have been refunded and the total amount has been added back to your account balance.
      
      Market Refund Details:
      - Total Refund Amount: $${amount.toFixed(2)}
      - Market ID: ${marketId}
      - Number of Bets Refunded: ${betCount}
      
      All your bets in this market have been refunded. The total amount is now available in your account balance and can be used for new bets or withdrawals.
      
      View Your Wallet: ${process.env.NEXT_PUBLIC_SITE_URL || 'https://betfusionzim.com'}/wallet
      
      If you have any questions about this market refund, please contact our support team.
      
      © 2024 BetFusion Zim. All rights reserved.
      This is an automated message. Please do not reply to this email.
    `
    
    return { subject, html, text }
  }
}

// Default email configuration
export const defaultEmailConfig: EmailConfig = {
  host: process.env.SMTP_HOST || 'smtp.gmail.com',
  port: parseInt(process.env.SMTP_PORT || '587'),
  secure: process.env.SMTP_SECURE === 'true',
  auth: {
    user: process.env.SMTP_USER || '',
    pass: process.env.SMTP_PASS || ''
  }
}

// Create default email service instance
export const emailService = new EmailService(defaultEmailConfig)
