import { supabase } from '@/lib/supabase-client'
import { detectDevice } from '@/lib/utils/deviceDetection'
import { checkSupabaseEnv } from '@/lib/utils/envCheck'

export interface LoginSessionData {
  user_id: string
  session_token: string
  ip_address?: string
  user_agent?: string
  device_type?: string
  browser?: string
  os?: string
  location?: string
}

export class LoginTrackingService {
  private static instance: LoginTrackingService
  private currentSessionId: string | null = null

  static getInstance(): LoginTrackingService {
    if (!LoginTrackingService.instance) {
      LoginTrackingService.instance = new LoginTrackingService()
    }
    return LoginTrackingService.instance
  }

  async trackLogin(userId: string, sessionToken: string, ipAddress?: string): Promise<string | null> {
    try {
      console.log('LoginTrackingService: Starting trackLogin...', { userId, sessionToken })
      
      // Check if environment variables are set
      const envCheck = checkSupabaseEnv()
      if (!envCheck.isValid) {
        console.warn('LoginTrackingService: Supabase environment variables not configured. Login tracking disabled.')
        console.warn('LoginTrackingService:', envCheck.message)
        return null
      }
      
      // First, test if we can connect to Supabase
      console.log('LoginTrackingService: Testing Supabase connection...')
      const { data: testData, error: testError } = await supabase
        .from('profiles')
        .select('id')
        .eq('id', userId)
        .limit(1)

      if (testError) {
        console.error('LoginTrackingService: Supabase connection test failed:', {
          error: testError,
          message: testError.message,
          details: testError.details,
          hint: testError.hint,
          code: testError.code
        })
        return null
      }

      console.log('LoginTrackingService: Supabase connection test successful')

      const deviceInfo = detectDevice()
      
      const sessionData: LoginSessionData = {
        user_id: userId,
        session_token: sessionToken,
        ip_address: ipAddress,
        user_agent: deviceInfo.userAgent,
        device_type: deviceInfo.deviceType,
        browser: deviceInfo.browser,
        os: deviceInfo.os,
        location: await this.getLocationFromIP(ipAddress)
      }

      console.log('LoginTrackingService: Attempting to track login session...', { userId, sessionToken })

      const { data, error } = await supabase
        .from('login_sessions')
        .insert(sessionData)
        .select()
        .single()

      if (error) {
        console.error('Error tracking login:', {
          error,
          message: error.message,
          details: error.details,
          hint: error.hint,
          code: error.code,
          timestamp: new Date().toISOString()
        })
        
        // If the table doesn't exist, log a helpful message
        if (error.code === 'PGRST116' || error.message?.includes('relation "login_sessions" does not exist') || error.message?.includes('does not exist')) {
          console.warn('LoginTrackingService: login_sessions table does not exist. Please run the create-login-sessions-table.sql script in your Supabase database.')
          console.warn('LoginTrackingService: This is expected if you haven\'t created the login_sessions table yet.')
          return null
        }
        
        // If we get an empty error object, it might be a connection issue
        if (!error.message && !error.code && !error.details) {
          console.warn('LoginTrackingService: Received empty error object. This usually indicates a Supabase connection issue.')
          console.warn('LoginTrackingService: Please check your environment variables and Supabase configuration.')
          return null
        }
        
        return null
      }

      console.log('LoginTrackingService: Login session tracked successfully:', data.id)
      this.currentSessionId = data.id
      return data.id
    } catch (error) {
      console.error('Error in trackLogin:', error)
      return null
    }
  }

  async updateActivity(sessionId?: string): Promise<boolean> {
    const targetSessionId = sessionId || this.currentSessionId
    if (!targetSessionId) return false

    try {
      const { error } = await supabase
        .from('login_sessions')
        .update({ last_activity: new Date().toISOString() })
        .eq('id', targetSessionId)
        .eq('is_active', true)

      if (error) {
        console.error('Error updating session activity:', {
          error,
          message: error.message,
          details: error.details,
          hint: error.hint,
          code: error.code,
          timestamp: new Date().toISOString()
        })
        return false
      }

      return true
    } catch (error) {
      console.error('Error in updateActivity:', error)
      return false
    }
  }

  async trackLogout(sessionId?: string): Promise<boolean> {
    const targetSessionId = sessionId || this.currentSessionId
    if (!targetSessionId) return false

    try {
      const { error } = await supabase
        .from('login_sessions')
        .update({
          is_active: false,
          logout_time: new Date().toISOString()
        })
        .eq('id', targetSessionId)

      if (error) {
        console.error('Error tracking logout:', {
          error,
          message: error.message,
          details: error.details,
          hint: error.hint,
          code: error.code,
          timestamp: new Date().toISOString()
        })
        return false
      }

      if (targetSessionId === this.currentSessionId) {
        this.currentSessionId = null
      }

      return true
    } catch (error) {
      console.error('Error in trackLogout:', error)
      return false
    }
  }

  async endAllSessions(userId: string): Promise<boolean> {
    try {
      const { error } = await supabase
        .from('login_sessions')
        .update({
          is_active: false,
          logout_time: new Date().toISOString()
        })
        .eq('user_id', userId)
        .eq('is_active', true)

      if (error) {
        console.error('Error ending all sessions:', {
          error,
          message: error.message,
          details: error.details,
          hint: error.hint,
          code: error.code,
          timestamp: new Date().toISOString()
        })
        return false
      }

      this.currentSessionId = null
      return true
    } catch (error) {
      console.error('Error in endAllSessions:', error)
      return false
    }
  }

  private async getLocationFromIP(ipAddress?: string): Promise<string> {
    if (!ipAddress) return 'Unknown Location'
    
    try {
      // Skip localhost and private IPs
      if (ipAddress === '127.0.0.1' || ipAddress === '::1' || ipAddress.startsWith('192.168.') || ipAddress.startsWith('10.') || ipAddress.startsWith('172.')) {
        return 'Local Network'
      }
      
      console.log('LoginTrackingService: Fetching location for IP:', ipAddress)
      
      // Use ipapi.co free service (1000 requests/day)
      const response = await fetch(`https://ipapi.co/${ipAddress}/json/`, {
        method: 'GET',
        headers: {
          'User-Agent': 'BetFusion-Zim/1.0'
        }
      })
      
      if (!response.ok) {
        console.warn('LoginTrackingService: IP geolocation service unavailable')
        return 'Unknown Location'
      }
      
      const data = await response.json()
      
      if (data.error) {
        console.warn('LoginTrackingService: IP geolocation error:', data.reason)
        return 'Unknown Location'
      }
      
      // Format location string
      const locationParts = []
      if (data.city) locationParts.push(data.city)
      if (data.region) locationParts.push(data.region)
      if (data.country_name) locationParts.push(data.country_name)
      
      const location = locationParts.length > 0 ? locationParts.join(', ') : 'Unknown Location'
      console.log('LoginTrackingService: Location resolved:', location)
      
      return location
    } catch (error) {
      console.error('LoginTrackingService: Error getting location from IP:', error)
      return 'Unknown Location'
    }
  }

  getCurrentSessionId(): string | null {
    return this.currentSessionId
  }

  setCurrentSessionId(sessionId: string): void {
    this.currentSessionId = sessionId
  }
}

// Export singleton instance
export const loginTracking = LoginTrackingService.getInstance()

// Utility function to get client IP (for server-side usage)
export async function getClientIP(request: Request): Promise<string | undefined> {
  // Try various headers that might contain the real IP
  const forwardedFor = request.headers.get('x-forwarded-for')
  const realIP = request.headers.get('x-real-ip')
  const cfConnectingIP = request.headers.get('cf-connecting-ip')
  
  if (forwardedFor) {
    return forwardedFor.split(',')[0].trim()
  }
  
  if (realIP) {
    return realIP
  }
  
  if (cfConnectingIP) {
    return cfConnectingIP
  }
  
  return undefined
}
