// Notification Service for BetFusion Zim
// Handles real-time notifications for various events

import { supabase } from '@/lib/supabase-client'
import { emailService } from './emailService'

export interface NotificationData {
  id: string
  user_id: string
  title: string
  message: string
  type: 'info' | 'success' | 'warning' | 'error'
  is_read: boolean
  data?: Record<string, any>
  created_at: string
  updated_at: string
}

export type NotificationType = 
  | 'bet_won'
  | 'bet_lost' 
  | 'new_market_available'
  | 'market_resolved'
  | 'deposit_received'
  | 'withdrawal_processed'
  | 'referral_bonus'
  | 'achievement_unlocked'
  | 'price_alert'
  | 'liquidity_reward'
  | 'market_seeded'
  | 'kyc_approved'
  | 'kyc_rejected'

export class NotificationService {
  /**
   * Create a new notification
   */
  static async createNotification(
    userId: string,
    title: string,
    message: string,
    type: NotificationData['type'] = 'info',
    data?: Record<string, any>,
    sendEmail: boolean = false
  ): Promise<NotificationData | null> {
    try {
      const { data: notification, error } = await supabase
        .from('notifications')
        .insert({
          user_id: userId,
          title,
          message,
          type,
          data,
          is_read: false
        })
        .select()
        .single()

      if (error) {
        console.error('Error creating notification:', error)
        return null
      }

      // Send email if requested
      if (sendEmail) {
        await this.sendNotificationEmail(userId, title, message, type, data)
      }

      return notification
    } catch (error) {
      console.error('Error in createNotification:', error)
      return null
    }
  }

  /**
   * Get user notifications
   */
  static async getUserNotifications(
    userId: string,
    limit: number = 20,
    unreadOnly: boolean = false
  ): Promise<NotificationData[]> {
    try {
      let query = supabase
        .from('notifications')
        .select('*')
        .eq('user_id', userId)
        .order('created_at', { ascending: false })
        .limit(limit)

      if (unreadOnly) {
        query = query.eq('is_read', false)
      }

      const { data, error } = await query

      if (error) {
        console.error('Error fetching notifications:', error)
        return []
      }

      return data || []
    } catch (error) {
      console.error('Error in getUserNotifications:', error)
      return []
    }
  }

  /**
   * Mark notification as read
   */
  static async markAsRead(notificationId: string): Promise<boolean> {
    try {
      const { error } = await supabase
        .from('notifications')
        .update({ is_read: true, updated_at: new Date().toISOString() })
        .eq('id', notificationId)

      if (error) {
        console.error('Error marking notification as read:', error)
        return false
      }

      return true
    } catch (error) {
      console.error('Error in markAsRead:', error)
      return false
    }
  }

  /**
   * Mark all notifications as read for a user
   */
  static async markAllAsRead(userId: string): Promise<boolean> {
    try {
      const { error } = await supabase
        .from('notifications')
        .update({ is_read: true, updated_at: new Date().toISOString() })
        .eq('user_id', userId)
        .eq('is_read', false)

      if (error) {
        console.error('Error marking all notifications as read:', error)
        return false
      }

      return true
    } catch (error) {
      console.error('Error in markAllAsRead:', error)
      return false
    }
  }

  /**
   * Get unread notification count
   */
  static async getUnreadCount(userId: string): Promise<number> {
    try {
      const { count, error } = await supabase
        .from('notifications')
        .select('*', { count: 'exact', head: true })
        .eq('user_id', userId)
        .eq('is_read', false)

      if (error) {
        console.error('Error getting unread count:', error)
        return 0
      }

      return count || 0
    } catch (error) {
      console.error('Error in getUnreadCount:', error)
      return 0
    }
  }

  /**
   * Notification templates for different events
   */
  static getNotificationTemplate(
    type: NotificationType,
    data?: Record<string, any>
  ): { title: string; message: string; notificationType: NotificationData['type'] } {
    switch (type) {
      case 'bet_won':
        return {
          title: '🎉 Bet Won!',
          message: `Congratulations! You won $${data?.amount?.toFixed(2) || '0.00'} on "${data?.marketTitle || 'your bet'}"`,
          notificationType: 'success'
        }

      case 'bet_lost':
        return {
          title: 'Bet Result',
          message: `Your bet on "${data?.marketTitle || 'the market'}" didn't win this time. Better luck next time!`,
          notificationType: 'info'
        }

      case 'new_market_available':
        return {
          title: '🆕 New Market Available',
          message: `"${data?.marketTitle || 'New market'}" is now live for betting!`,
          notificationType: 'info'
        }

      case 'market_resolved':
        return {
          title: 'Market Resolved',
          message: `"${data?.marketTitle || 'Market'}" has been resolved. Check your results!`,
          notificationType: 'info'
        }

      case 'deposit_received':
        return {
          title: '💰 Deposit Received',
          message: `Your deposit of $${data?.amount?.toFixed(2) || '0.00'} has been processed successfully.`,
          notificationType: 'success'
        }

      case 'withdrawal_processed':
        return {
          title: '💸 Withdrawal Processed',
          message: `Your withdrawal of $${data?.amount?.toFixed(2) || '0.00'} has been processed.`,
          notificationType: 'success'
        }

      case 'referral_bonus':
        return {
          title: '🎁 Referral Bonus',
          message: `You received a $${data?.amount?.toFixed(2) || '0.00'} bonus for referring a friend!`,
          notificationType: 'success'
        }

      case 'achievement_unlocked':
        return {
          title: '🏆 Achievement Unlocked',
          message: `You've unlocked the "${data?.achievementName || 'Achievement'}" achievement!`,
          notificationType: 'success'
        }

      case 'price_alert':
        return {
          title: '📈 Price Alert',
          message: `"${data?.marketTitle || 'Market'}" price has reached your alert level of ${data?.price || '0'}%`,
          notificationType: 'warning'
        }

      case 'liquidity_reward':
        return {
          title: '💧 Liquidity Reward',
          message: `You earned $${data?.amount?.toFixed(2) || '0.00'} for providing liquidity to "${data?.marketTitle || 'market'}"`,
          notificationType: 'success'
        }

      case 'market_seeded':
        return {
          title: '🌱 Market Seeded',
          message: `Your market "${data?.marketTitle || 'Market'}" has been successfully seeded with $${data?.amount?.toFixed(2) || '0.00'}`,
          notificationType: 'success'
        }

      case 'kyc_approved':
        return {
          title: '✅ KYC Approved',
          message: 'Your identity verification has been approved. You now have full access to all features.',
          notificationType: 'success'
        }

      case 'kyc_rejected':
        return {
          title: '❌ KYC Rejected',
          message: `Your identity verification was rejected. Reason: ${data?.reason || 'Please try again'}`,
          notificationType: 'error'
        }

      default:
        return {
          title: 'Notification',
          message: 'You have a new notification',
          notificationType: 'info'
        }
    }
  }

  /**
   * Create notification for specific event type
   */
  static async createEventNotification(
    userId: string,
    eventType: NotificationType,
    eventData?: Record<string, any>,
    sendEmail: boolean = false
  ): Promise<NotificationData | null> {
    const template = this.getNotificationTemplate(eventType, eventData)
    
    return this.createNotification(
      userId,
      template.title,
      template.message,
      template.notificationType,
      eventData,
      sendEmail
    )
  }

  /**
   * Bulk create notifications for multiple users
   */
  static async createBulkNotifications(
    userIds: string[],
    eventType: NotificationType,
    eventData?: Record<string, any>
  ): Promise<NotificationData[]> {
    const template = this.getNotificationTemplate(eventType, eventData)
    const notifications: NotificationData[] = []

    for (const userId of userIds) {
      const notification = await this.createNotification(
        userId,
        template.title,
        template.message,
        template.notificationType,
        eventData
      )
      
      if (notification) {
        notifications.push(notification)
      }
    }

    return notifications
  }

  /**
   * Send notification email to user
   */
  private static async sendNotificationEmail(
    userId: string,
    title: string,
    message: string,
    type: NotificationData['type'],
    data?: Record<string, any>
  ): Promise<void> {
    try {
      // Get user profile to get email and name
      const { data: profile, error: profileError } = await supabase
        .from('profiles')
        .select('email, first_name, last_name')
        .eq('id', userId)
        .single()

      if (profileError || !profile?.email) {
        console.error('Error fetching user profile for email:', profileError)
        return
      }

      const userName = profile.first_name || 'User'
      const userEmail = profile.email

      // Send appropriate email based on notification type
      switch (type) {
        case 'success':
          if (data?.betResult) {
            await emailService.sendBetResultEmail(
              userEmail,
              userName,
              data.betResult.outcome,
              data.betResult.amount,
              data.betResult.marketTitle
            )
          } else if (data?.deposit) {
            await emailService.sendDepositConfirmationEmail(
              userEmail,
              userName,
              data.deposit.amount,
              data.deposit.transactionId
            )
          } else if (data?.withdrawal) {
            await emailService.sendWithdrawalConfirmationEmail(
              userEmail,
              userName,
              data.withdrawal.amount,
              data.withdrawal.transactionId
            )
          } else if (data?.kycStatus) {
            await emailService.sendKYCStatusEmail(
              userEmail,
              userName,
              data.kycStatus.status,
              data.kycStatus.reason
            )
          }
          break

        case 'info':
          if (data?.marketResolution) {
            await emailService.sendMarketResolutionEmail(
              userEmail,
              userName,
              data.marketResolution.marketTitle,
              data.marketResolution.winningOutcome,
              data.marketResolution.userOutcome,
              data.marketResolution.won,
              data.marketResolution.amount
            )
          }
          break

        default:
          // Send generic notification email
          await emailService.sendEmail({
            to: userEmail,
            subject: title,
            html: `
              <div style="font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto;">
                <h2>${title}</h2>
                <p>${message}</p>
                <p>Best regards,<br>The BetFusion Zim Team</p>
              </div>
            `,
            text: `${title}\n\n${message}\n\nBest regards,\nThe BetFusion Zim Team`
          })
          break
      }
    } catch (error) {
      console.error('Error sending notification email:', error)
    }
  }

  /**
   * Send welcome email to new user
   */
  static async sendWelcomeEmail(userId: string): Promise<boolean> {
    try {
      const { data: profile, error } = await supabase
        .from('profiles')
        .select('email, first_name')
        .eq('id', userId)
        .single()

      if (error || !profile?.email) {
        console.error('Error fetching user profile for welcome email:', error)
        return false
      }

      return await emailService.sendWelcomeEmail(
        profile.email,
        profile.first_name || 'User'
      )
    } catch (error) {
      console.error('Error sending welcome email:', error)
      return false
    }
  }
}

export default NotificationService
