// Notification Triggers Service
// This service automatically triggers notifications for various events

import { NotificationService } from './notificationService'
import { supabase } from '@/lib/supabase-client'

export class NotificationTriggers {
  /**
   * Trigger notification when a bet is won
   */
  static async onBetWon(userId: string, marketId: string, amount: number, marketTitle: string) {
    return NotificationService.createEventNotification(userId, 'bet_won', {
      betResult: {
        outcome: 'won',
        amount,
        marketTitle
      }
    }, true) // Send email
  }

  /**
   * Trigger notification when a bet is lost
   */
  static async onBetLost(userId: string, marketId: string, amount: number, marketTitle: string) {
    return NotificationService.createEventNotification(userId, 'bet_lost', {
      betResult: {
        outcome: 'lost',
        amount,
        marketTitle
      }
    }, true) // Send email
  }

  /**
   * Trigger notification when a new market is created
   */
  static async onNewMarketCreated(marketId: string, marketTitle: string, createdBy?: string) {
    try {
      // Get users who might be interested in this market
      const { data: interestedUsers } = await supabase
        .from('profiles')
        .select('id')
        .eq('is_active', true)
        .not('id', 'eq', createdBy || '')

      if (interestedUsers && interestedUsers.length > 0) {
        const userIds = interestedUsers.map(user => user.id)
        return NotificationService.createBulkNotifications(userIds, 'new_market_available', {
          marketId,
          marketTitle
        })
      }
    } catch (error) {
      console.error('Error triggering new market notification:', error)
    }
  }

  /**
   * Trigger notification when a market is resolved
   */
  static async onMarketResolved(marketId: string, marketTitle: string, outcome: string) {
    try {
      // Get all users who have bets on this market
      const { data: bettors } = await supabase
        .from('bets')
        .select('user_id')
        .eq('market_id', marketId)

      if (bettors && bettors.length > 0) {
        const userIds = [...new Set(bettors.map(bet => bet.user_id))]
        return NotificationService.createBulkNotifications(userIds, 'market_resolved', {
          marketId,
          marketTitle,
          outcome
        })
      }
    } catch (error) {
      console.error('Error triggering market resolved notification:', error)
    }
  }

  /**
   * Trigger notification when a deposit is received
   */
  static async onDepositReceived(userId: string, amount: number, method: string, transactionId?: string) {
    return NotificationService.createEventNotification(userId, 'deposit_received', {
      deposit: {
        amount,
        method,
        transactionId: transactionId || `deposit_${Date.now()}`
      }
    }, true) // Send email
  }

  /**
   * Trigger notification when a withdrawal is processed
   */
  static async onWithdrawalProcessed(userId: string, amount: number, status: string, transactionId?: string) {
    const eventType = status === 'completed' ? 'withdrawal_processed' : 'error'
    return NotificationService.createEventNotification(userId, eventType, {
      withdrawal: {
        amount,
        status,
        transactionId: transactionId || `withdrawal_${Date.now()}`
      }
    }, true) // Send email
  }

  /**
   * Trigger notification when KYC is approved
   */
  static async onKYCApproved(userId: string) {
    return NotificationService.createEventNotification(userId, 'kyc_approved', {
      kycStatus: {
        status: 'approved'
      }
    }, true) // Send email
  }

  /**
   * Trigger notification when KYC is rejected
   */
  static async onKYCRejected(userId: string, reason: string) {
    return NotificationService.createEventNotification(userId, 'kyc_rejected', {
      kycStatus: {
        status: 'rejected',
        reason
      }
    }, true) // Send email
  }

  /**
   * Trigger notification when user achieves something
   */
  static async onAchievementUnlocked(userId: string, achievementName: string, points: number) {
    return NotificationService.createEventNotification(userId, 'achievement_unlocked', {
      achievementName,
      points
    })
  }

  /**
   * Trigger notification when user receives referral bonus
   */
  static async onReferralBonus(userId: string, amount: number, referredUser: string) {
    return NotificationService.createEventNotification(userId, 'referral_bonus', {
      amount,
      referredUser
    })
  }

  /**
   * Trigger notification when market is seeded
   */
  static async onMarketSeeded(userId: string, marketId: string, marketTitle: string, amount: number) {
    return NotificationService.createEventNotification(userId, 'market_seeded', {
      marketId,
      marketTitle,
      amount
    })
  }

  /**
   * Trigger notification for liquidity rewards
   */
  static async onLiquidityReward(userId: string, marketId: string, marketTitle: string, amount: number) {
    return NotificationService.createEventNotification(userId, 'liquidity_reward', {
      marketId,
      marketTitle,
      amount
    })
  }

  /**
   * Trigger notification for price alerts
   */
  static async onPriceAlert(userId: string, marketId: string, marketTitle: string, price: number, threshold: number) {
    return NotificationService.createEventNotification(userId, 'price_alert', {
      marketId,
      marketTitle,
      price,
      threshold
    })
  }

  /**
   * Process bet results and trigger appropriate notifications
   */
  static async processBetResults(marketId: string, winningOutcome: string) {
    try {
      // Get all bets for this market
      const { data: bets } = await supabase
        .from('bets')
        .select(`
          *,
          markets (
            id,
            title
          )
        `)
        .eq('market_id', marketId)

      if (!bets) return

      // Get market info
      const { data: market } = await supabase
        .from('markets')
        .select('title')
        .eq('id', marketId)
        .single()

      const marketTitle = market?.title || 'Market'

      // Process each bet
      for (const bet of bets) {
        const isWinning = bet.outcome === winningOutcome
        
        if (isWinning) {
          await this.onBetWon(bet.user_id, marketId, bet.amount, marketTitle)
        } else {
          await this.onBetLost(bet.user_id, marketId, bet.amount, marketTitle)
        }
      }

      // Notify all bettors that market is resolved
      await this.onMarketResolved(marketId, marketTitle, winningOutcome)
    } catch (error) {
      console.error('Error processing bet results:', error)
    }
  }

  /**
   * Setup automatic triggers for various database events
   */
  static async setupAutomaticTriggers() {
    // This would typically be set up in your database triggers
    // or webhook handlers for external services
    console.log('Setting up automatic notification triggers...')
  }
}

export default NotificationTriggers
