import { supabase } from './supabase-client'

export interface UploadResult {
  success: boolean
  url?: string
  error?: string
}

export interface UploadOptions {
  bucket: 'avatars' | 'kyc-documents'
  userId: string
  file: File
  folder?: string
  expiresIn?: number // seconds
}

/**
 * Upload a file to Supabase Storage with proper error handling
 */
export async function uploadFile(options: UploadOptions): Promise<UploadResult> {
  try {
    const { bucket, userId, file, folder = '', expiresIn = 60 * 60 * 24 * 7 } = options

    // Validate file
    const validation = validateFile(file, bucket)
    if (!validation.success) {
      return { success: false, error: validation.error }
    }

    // Create unique filename - ensure userId is the first folder for RLS policies
    const fileExt = file.name.split('.').pop()
    const timestamp = Date.now()
    const cleanFileName = file.name.replace(/[^a-zA-Z0-9.-]/g, '_')
    const fileName = `${userId}/${timestamp}_${cleanFileName}`

    console.log('Uploading file:', {
      bucket,
      userId,
      fileName,
      fileSize: file.size,
      fileType: file.type
    })

    // Upload file
    const { data: uploadData, error: uploadError } = await supabase.storage
      .from(bucket)
      .upload(fileName, file, {
        cacheControl: '3600',
        upsert: false
      })

    if (uploadError) {
      console.error('Upload error details:', {
        error: uploadError,
        bucket,
        fileName,
        userId
      })
      return { success: false, error: `Failed to upload file: ${uploadError.message}` }
    }

    // Get signed URL
    const { data: signedUrlData, error: urlError } = await supabase.storage
      .from(bucket)
      .createSignedUrl(fileName, expiresIn)

    if (urlError) {
      console.error('Signed URL error:', urlError)
      return { success: false, error: `Failed to get file URL: ${urlError.message}` }
    }

    return {
      success: true,
      url: signedUrlData.signedUrl
    }

  } catch (error) {
    console.error('Upload error:', error)
    return {
      success: false,
      error: error instanceof Error ? error.message : 'Unknown upload error'
    }
  }
}

/**
 * Delete a file from Supabase Storage
 */
export async function deleteFile(bucket: string, filePath: string): Promise<UploadResult> {
  try {
    const { error } = await supabase.storage
      .from(bucket)
      .remove([filePath])

    if (error) {
      console.error('Delete error:', error)
      return { success: false, error: `Failed to delete file: ${error.message}` }
    }

    return { success: true }
  } catch (error) {
    console.error('Delete error:', error)
    return {
      success: false,
      error: error instanceof Error ? error.message : 'Unknown delete error'
    }
  }
}

/**
 * Get a signed URL for an existing file
 */
export async function getSignedUrl(bucket: string, filePath: string, expiresIn: number = 60 * 60 * 24 * 7): Promise<UploadResult> {
  try {
    const { data, error } = await supabase.storage
      .from(bucket)
      .createSignedUrl(filePath, expiresIn)

    if (error) {
      console.error('Signed URL error:', error)
      return { success: false, error: `Failed to get file URL: ${error.message}` }
    }

    return {
      success: true,
      url: data.signedUrl
    }
  } catch (error) {
    console.error('Signed URL error:', error)
    return {
      success: false,
      error: error instanceof Error ? error.message : 'Unknown URL error'
    }
  }
}

/**
 * Validate file before upload
 */
function validateFile(file: File, bucket: 'avatars' | 'kyc-documents'): { success: boolean; error?: string } {
  if (bucket === 'avatars') {
    // Avatar validation
    if (file.size > 2 * 1024 * 1024) { // 2MB
      return { success: false, error: 'Avatar file size must be less than 2MB' }
    }
    if (!file.type.startsWith('image/')) {
      return { success: false, error: 'Avatar must be an image file' }
    }
  } else if (bucket === 'kyc-documents') {
    // KYC document validation
    if (file.size > 5 * 1024 * 1024) { // 5MB
      return { success: false, error: 'Document file size must be less than 5MB' }
    }
    const allowedTypes = ['image/jpeg', 'image/jpg', 'image/png', 'application/pdf']
    if (!allowedTypes.includes(file.type)) {
      return { success: false, error: 'Document must be a JPG, PNG, or PDF file' }
    }
  }

  return { success: true }
}

/**
 * Extract file path from signed URL for deletion
 */
export function extractFilePathFromUrl(url: string): string | null {
  try {
    const urlObj = new URL(url)
    const pathParts = urlObj.pathname.split('/')
    // Extract path after bucket name
    const bucketIndex = pathParts.findIndex(part => part === 'avatars' || part === 'kyc-documents')
    if (bucketIndex === -1 || bucketIndex === pathParts.length - 1) {
      return null
    }
    return pathParts.slice(bucketIndex + 1).join('/')
  } catch {
    return null
  }
}
