export interface Market {
  id: string
  title: string
  description: string
  category: string
  end_date: string
  image_url: string | null
  creator_id: string
  status: 'pending' | 'active' | 'paused' | 'closed' | 'resolved' | 'rejected' | 'cancelled' | 'refunded'
  outcome: string | null // Can be any outcome name now
  resolution_date: string | null
  total_volume: number
  total_liquidity: number // Total liquidity in the market
  market_type: 'binary' | 'multi_outcome' // Binary (yes/no) or multiple outcomes
  outcomes?: MarketOutcome[] // For multi-outcome markets
  yes_price?: number // For backward compatibility with binary markets
  no_price?: number // For backward compatibility with binary markets
  yes_pool?: number
  no_pool?: number
  seeded_volume?: number
  live_volume?: number
  creator_earnings?: number
  admin_fee?: number
  created_at: string
  updated_at: string
  // Dynamic fields for live updates
  shares_sold?: number
  total_shares?: number
}

export interface MarketOutcome {
  id: string
  market_id: string
  name: string
  description?: string
  price: number // Current price (0-1)
  pool: number // Total liquidity in this outcome
  volume: number // Total volume traded
  color?: string // Hex color code for visual representation
  image_url?: string // URL of the image to represent the outcome
  icon?: string // Icon/emoji to represent the outcome
  sort_order?: number // Order in which outcomes should be displayed
  is_resolved: boolean
  created_at: string
  updated_at: string
}

export interface Bet {
  id: string
  user_id: string
  market_id: string
  outcome: string // Can be 'yes', 'no', or any outcome name
  outcome_id?: string // Reference to specific outcome for multi-outcome markets
  amount: number
  price: number
  shares: number
  status: 'seeded' | 'live' | 'matched' | 'cancelled' | 'won' | 'lost'
  seeded_amount?: number
  live_amount?: number
  liquidity_contribution?: number // Amount contributed to liquidity
  created_at: string
  updated_at: string
}

export interface Transaction {
  id: string
  user_id: string
  type: 'deposit' | 'withdrawal' | 'bet' | 'win' | 'refund' | 'fee' | 'bonus' | 'transfer_in' | 'transfer_out' | 'market_creation' | 'seeding' | 'liquidity'
  status: 'pending' | 'processing' | 'completed' | 'failed' | 'cancelled'
  amount: number
  currency: string
  description: string | null
  market_id: string | null
  bet_id: string | null
  payment_method_id: string | null
  external_transaction_id: string | null
  fee_amount: number
  net_amount: number
  metadata: Record<string, any> | null
  processed_at: string | null
  created_at: string
  updated_at: string
  payment_method?: PaymentMethod
}

export interface Profile {
  id: string
  email: string
  username: string | null
  avatar_url: string | null
  balance: number
  is_admin: boolean
  first_name: string | null
  last_name: string | null
  phone: string | null
  date_of_birth: string | null
  country: string | null
  timezone: string | null
  kyc_verified: boolean
  kyc_verified_at: string | null
  kyc_status: 'pending' | 'approved' | 'rejected' | null
  kyc_documents: Record<string, string> | null
  kyc_submitted_at: string | null
  kyc_rejection_reason: string | null
  total_points: number
  referral_code: string | null
  wallet_address: string | null
  preferred_payment_method: string
  daily_deposit_limit: number
  daily_withdrawal_limit: number
  monthly_deposit_limit: number
  monthly_withdrawal_limit: number
  account_status: string | null
  last_login: string | null
  login_count: number | null
  bio: string | null
  created_at: string
  updated_at: string
}

export interface PaymentMethod {
  id: string
  user_id: string
  type: 'credit_card' | 'bank_account' | 'crypto_wallet' | 'paypal' | 'paynow_zim' | 'pesapal'
  provider: string
  provider_account_id: string | null
  last_four: string | null
  brand: string | null
  is_default: boolean
  is_verified: boolean
  currency: 'USD' // USD-only implementation
  metadata: Record<string, any> | null
  created_at: string
  updated_at: string
}

export interface WalletTransaction {
  id: string
  user_id: string
  type: 'deposit' | 'withdrawal' | 'transfer_in' | 'transfer_out' | 'fee' | 'bonus' | 'bet' | 'win' | 'market_creation' | 'seeding' | 'liquidity'
  status: 'pending' | 'processing' | 'completed' | 'failed' | 'cancelled'
  amount: number
  currency: string
  payment_method_id: string | null
  external_transaction_id: string | null
  description: string | null
  fee_amount: number
  net_amount: number
  metadata: Record<string, any> | null
  processed_at: string | null
  created_at: string
  updated_at: string
  payment_method?: PaymentMethod
  market_id?: string
  bet_id?: string
}

export interface WithdrawalRequest {
  id: string
  user_id: string
  amount: number
  payment_method_id: string
  status: 'pending' | 'approved' | 'rejected' | 'processing' | 'completed'
  admin_notes: string | null
  rejection_reason: string | null
  processed_by: string | null
  processed_at: string | null
  created_at: string
  updated_at: string
  payment_method?: PaymentMethod
}

export interface DailyLimits {
  id: string
  user_id: string
  date: string
  deposits_count: number
  deposits_amount: number
  withdrawals_count: number
  withdrawals_amount: number
  created_at: string
  updated_at: string
}

export interface Reward {
  id: string
  user_id: string
  type: 'betting' | 'referral' | 'achievement' | 'daily_login'
  points: number
  description: string | null
  metadata: Record<string, any> | null
  created_at: string
}

export interface Achievement {
  id: string
  name: string
  description: string
  icon: string | null
  points: number
  criteria: Record<string, any>
  category: 'betting' | 'social' | 'milestone'
  created_at: string
}

export interface UserAchievement {
  id: string
  user_id: string
  achievement_id: string
  unlocked_at: string
  achievement?: Achievement
}

export interface KycVerification {
  id: string
  user_id: string
  status: 'pending' | 'approved' | 'rejected' | 'expired'
  document_type: 'passport' | 'drivers_license' | 'national_id'
  document_number: string | null
  document_front_url: string | null
  document_back_url: string | null
  selfie_url: string | null
  verified_at: string | null
  rejected_reason: string | null
  metadata: Record<string, any> | null
  created_at: string
  updated_at: string
}

export interface Referral {
  id: string
  referrer_id: string
  referee_id: string
  referral_code: string
  status: 'pending' | 'completed' | 'expired'
  reward_points: number
  created_at: string
}

export interface LeaderboardEntry {
  id: string
  username: string | null
  email: string
  total_points: number
  total_winnings: number
  total_bet_amount: number
  markets_traded: number
  winning_bets: number
  total_bets: number
  win_rate: number
  kyc_verified: boolean
  profit_rank: number
  volume_rank: number
  accuracy_rank: number
  points_rank: number
}

export interface MarketWithBets extends Market {
  bets: Bet[]
}

export interface UserPortfolio {
  profile: Profile
  total_bets: number
  total_winnings: number
  active_bets: Bet[]
  transaction_history: Transaction[]
}

export interface BetWithMarket extends Bet {
  markets?: {
    title: string
    status: string
    outcome: string | null
  }
}

export interface TransactionWithMarket extends Transaction {
  markets?: {
    title: string
  }
}

export interface Stats {
  total_markets: number
  total_volume: number
  active_bets: number
  total_users: number
  total_profit: number
  win_rate: number
}

export interface MarketWithStats {
  id: string
  title: string
  description: string
  status: string
  admin_status: string
  outcome: string | null
  end_date: string
  total_volume: number
  yes_price: number
  no_price: number
  creator_id: string
  creator_earnings: number
  admin_fee: number
  created_at: string
  resolved_at: string | null
  admin_notes: string | null
  admin_action_at: string | null
  creator_email: string
  admin_username: string | null
  admin_status_display: string
  totalBets: number
  yesBets: number
  noBets: number
  seededVolume: number
  liveVolume: number
}
