// Device and browser detection utilities

export interface DeviceInfo {
  deviceType: 'desktop' | 'mobile' | 'tablet'
  browser: string
  os: string
  userAgent: string
}

export function detectDevice(): DeviceInfo {
  if (typeof window === 'undefined') {
    return {
      deviceType: 'desktop',
      browser: 'Unknown',
      os: 'Unknown',
      userAgent: ''
    }
  }

  const userAgent = navigator.userAgent
  const userAgentLower = userAgent.toLowerCase()

  // Detect device type
  let deviceType: 'desktop' | 'mobile' | 'tablet' = 'desktop'
  
  if (/tablet|ipad|playbook|silk/i.test(userAgent)) {
    deviceType = 'tablet'
  } else if (/mobile|iphone|ipod|android|blackberry|opera mini|iemobile|wpdesktop/i.test(userAgent)) {
    deviceType = 'mobile'
  }

  // Detect browser
  let browser = 'Unknown'
  if (userAgentLower.includes('chrome') && !userAgentLower.includes('edg')) {
    browser = 'Chrome'
  } else if (userAgentLower.includes('firefox')) {
    browser = 'Firefox'
  } else if (userAgentLower.includes('safari') && !userAgentLower.includes('chrome')) {
    browser = 'Safari'
  } else if (userAgentLower.includes('edg')) {
    browser = 'Edge'
  } else if (userAgentLower.includes('opera') || userAgentLower.includes('opr')) {
    browser = 'Opera'
  }

  // Detect OS
  let os = 'Unknown'
  if (userAgentLower.includes('windows')) {
    os = 'Windows'
  } else if (userAgentLower.includes('macintosh') || userAgentLower.includes('mac os x')) {
    os = 'macOS'
  } else if (userAgentLower.includes('linux')) {
    os = 'Linux'
  } else if (userAgentLower.includes('android')) {
    os = 'Android'
  } else if (userAgentLower.includes('iphone') || userAgentLower.includes('ipad')) {
    os = 'iOS'
  }

  return {
    deviceType,
    browser,
    os,
    userAgent
  }
}

export function getLocationFromIP(ip: string): Promise<string> {
  // This is a placeholder - in a real app, you'd use a geolocation service
  // For now, we'll return a generic location
  return Promise.resolve('Unknown Location')
}

export function formatSessionInfo(session: any): string {
  const { browser, os, deviceType } = session
  
  let info = browser || 'Unknown Browser'
  
  if (os && os !== 'Unknown') {
    info += ` on ${os}`
  }
  
  if (deviceType && deviceType !== 'desktop') {
    info += ` (${deviceType})`
  }
  
  return info
}

export function formatTimeAgo(dateString: string): string {
  const date = new Date(dateString)
  const now = new Date()
  const diffInSeconds = Math.floor((now.getTime() - date.getTime()) / 1000)
  
  if (diffInSeconds < 60) {
    return 'Just now'
  }
  
  const diffInMinutes = Math.floor(diffInSeconds / 60)
  if (diffInMinutes < 60) {
    return `${diffInMinutes} minute${diffInMinutes > 1 ? 's' : ''} ago`
  }
  
  const diffInHours = Math.floor(diffInMinutes / 60)
  if (diffInHours < 24) {
    return `${diffInHours} hour${diffInHours > 1 ? 's' : ''} ago`
  }
  
  const diffInDays = Math.floor(diffInHours / 24)
  if (diffInDays < 7) {
    return `${diffInDays} day${diffInDays > 1 ? 's' : ''} ago`
  }
  
  const diffInWeeks = Math.floor(diffInDays / 7)
  if (diffInWeeks < 4) {
    return `${diffInWeeks} week${diffInWeeks > 1 ? 's' : ''} ago`
  }
  
  const diffInMonths = Math.floor(diffInDays / 30)
  if (diffInMonths < 12) {
    return `${diffInMonths} month${diffInMonths > 1 ? 's' : ''} ago`
  }
  
  const diffInYears = Math.floor(diffInDays / 365)
  return `${diffInYears} year${diffInYears > 1 ? 's' : ''} ago`
}
