// Test script to check what columns exist in the bets table
const { createClient } = require('@supabase/supabase-js')

const supabaseUrl = process.env.NEXT_PUBLIC_SUPABASE_URL || 'your-supabase-url'
const supabaseKey = process.env.NEXT_PUBLIC_SUPABASE_ANON_KEY || 'your-supabase-key'

const supabase = createClient(supabaseUrl, supabaseKey)

async function testBetsTable() {
  console.log('🔍 Testing bets table structure...')
  
  try {
    // Try to get a sample bet to see what columns exist
    const { data: bets, error } = await supabase
      .from('bets')
      .select('*')
      .limit(1)
    
    if (error) {
      console.error('❌ Error fetching bets:', error)
      return
    }
    
    if (bets && bets.length > 0) {
      console.log('✅ Bets table accessible')
      console.log('Available columns:', Object.keys(bets[0]))
      console.log('Sample bet:', bets[0])
    } else {
      console.log('✅ Bets table accessible but empty')
      console.log('Trying to insert a test bet...')
      
      // Try to insert a minimal bet to see what columns are required
      const { data: testBet, error: insertError } = await supabase
        .from('bets')
        .insert({
          user_id: '00000000-0000-0000-0000-000000000000', // Dummy UUID
          market_id: '00000000-0000-0000-0000-000000000000', // Dummy UUID
          outcome: 'test',
          amount: 1.00,
          price: 0.5,
          shares: 2.0,
          status: 'matched'
        })
        .select()
        .single()
      
      if (insertError) {
        console.error('❌ Error inserting test bet:', insertError)
        console.error('Error details:', {
          message: insertError.message,
          details: insertError.details,
          hint: insertError.hint,
          code: insertError.code
        })
      } else {
        console.log('✅ Test bet inserted successfully:', testBet)
        
        // Clean up the test bet
        await supabase
          .from('bets')
          .delete()
          .eq('id', testBet.id)
        console.log('✅ Test bet cleaned up')
      }
    }
    
  } catch (error) {
    console.error('❌ Unexpected error:', error)
  }
}

testBetsTable()
